<?php
/*
 * RolUsuarioController.php
 * Springfield
 * Copyright ©: 2022 Integrasoft. Todos los derechos reservados.
 * Desarrollador(es):
 *      L.I. Humberto Javier Flores Tirado.
 * Fecha de creación: 25/03/2022.
 * Fecha de modificación: 25/03/2022.
 * Descripción:
 */

namespace app\controllers;

use Yii;
use yii\filters\AccessControl;
use yii\filters\VerbFilter;
use yii\helpers\ArrayHelper;
use yii\helpers\Url;
use yii\web\Controller;
use yii\web\Response;
use yii\widgets\ActiveForm;
use yii\db\IntegrityException;



use app\models\log\BitacoraUsuario;
use app\models\log\BitacoraSistema;

use app\models\rolusuario\RolUsuarioPermiso;
use app\models\permiso\Permiso;
use app\models\rolusuario\RolUsuario;
use app\models\rolusuario\RolUsuarioSearch;

/**
 * Clase controlador de perfiles.
 *
 * Class RolUsuarioController
 * @package app\controllers
 */
class RolUsuarioController extends Controller {
    private $controlador;
    private $logu;
    private $logs;

    /**
     * Método que restringue el acceso de las acciones de acuerdo a los roles.
     *
     * @return array
     */
    public function behaviors() {
        return [
            'access' => [
                'class' => AccessControl::className(),
                'only' => ['index', 'registrar', 'actualizar', 'ver', 'eliminar'],
                'rules' => [
                    [
                        'actions' => ['index', 'registrar', 'actualizar', 'ver', 'eliminar'],
                        'allow' => true,
                        'roles' => ['@'],
                        'matchCallback' => function($rule, $action) {
                            $permiso = null;

                            switch($action->id) {
                                case 'index':
                                    $permiso = 'Buscar rol de usuario';
                                    break;

                                case 'registrar':
                                    $permiso = 'Registrar rol de usuario';
                                    break;

                                case 'actualizar':
                                    $permiso = 'Actualizar rol de usuario';
                                    break;

                                case 'ver':
                                    $permiso = 'Ver rol de usuario';
                                    break;

                                case 'eliminar':
                                    $permiso = 'Eliminar rol de usuario';
                                    break;

                                default:
                                    return false;
                            }

                            if(isset(Yii::$app->user->identity))
                                return Yii::$app->user->identity->datos->verificarPermisos($permiso);
                            else
                                return false;
                        }
                    ],
                ],
            ],
            'verbs' => [
                'class' => VerbFilter::className(),
                'actions' => [
                    'logout' => ['post'],
                ],
            ],
        ];
    }

    /**
     * @inheritdoc
     */
    public function actions() {
        $this->layout = "dashboard";
        $this->controlador = "RolUsuarioController";
        $this->logu = new BitacoraUsuario();
        $this->logs = new BitacoraSistema();

        return [
            'error' => [
                'class' => 'yii\web\ErrorAction',
            ],
            'captcha' => [
                'class' => 'yii\captcha\CaptchaAction',
                'fixedVerifyCode' => YII_ENV_TEST ? 'testme' : null,
            ],

        ];
    }

    public function actionStatus() {}

    /**
     * Muestra la pantalla de gestión de roles de usuario.
     */
    public function actionIndex() {
        $searchModel = new RolUsuarioSearch();
        $dataProvider = $searchModel->search(Yii::$app->request->queryParams);

        return $this->render('index', [
            'searchModel' => $searchModel,
            'dataProvider' => $dataProvider
        ]);
    }

    /**
     * Muestra la pantalla de registro de un rol de usuario, además realiza el registro.
     *
     * @return array|string
     */
    public function actionRegistrar() {
        $model = new RolUsuario();

        $listaPermisos = ArrayHelper::map(Permiso::find()->all(), "idPermiso", "permiso");

        if($model->load(Yii::$app->request->post()) && Yii::$app->request->isAjax) {
            Yii::$app->response->format = Response::FORMAT_JSON;
            return ActiveForm::validate($model);
        }

        if($model->load(Yii::$app->request->post())) {
            $idUsuarioCreate = Yii::$app->user->identity->idUsuario;
            $idRolUsuarioCreate = Yii::$app->user->identity->rolUsuario->idRolUsuario;

            try {
                if(RolUsuario::verificarExistencia($model->rolUsuario))
                    throw new \Exception("Ya existe un rol de usuario registrado con ese nombre");

                $transaccion = Yii::$app->db->beginTransaction();
                $model->registrar($idUsuarioCreate, $idRolUsuarioCreate);
                $model->registrarPermisos($model->permisos, $idUsuarioCreate,
                    $idRolUsuarioCreate, $this->controlador, $this->logu, $this->logs);
                $transaccion->commit();

                $mensaje =  "SE REGISTRÓ EL ROL DE USUARIO CON ID: " . $model->idRolUsuario;
                $this->logu->registrar($idUsuarioCreate, $idRolUsuarioCreate, $mensaje, 1);
                $this->logs->registrar($idUsuarioCreate, $idRolUsuarioCreate, "DEBUG", $this->controlador, $mensaje);

                Yii::$app->session->setFlash('success', "Se ha registrado el rol de usuario");

                $model->rolUsuario = null;
                $model->permisos = null;
            } catch(\Exception $ex) {
                if(isset($transaccion))
                    $transaccion->rollBack();

                $mensaje =  "NO SE REGISTRÓ EL ROL DE USUARIO CON NOMBRE: " . $model->rolUsuario;
                $this->logu->registrar($idUsuarioCreate, $idRolUsuarioCreate, $mensaje, 0);
                $this->logs->registrar($idUsuarioCreate, $idRolUsuarioCreate, "ERROR", $this->controlador, $ex->getMessage());

                Yii::$app->session->setFlash('error', "Error al registrar el rol de usuario:<br />" . $ex->getMessage());
            }
        }

        return $this->render('registrar', [
            'model' => $model,
            'listaPermisos' => $listaPermisos,
        ]);
    }

    /**
     * Muestra la pantalla de actualización de un rol de usuario, además realiza la actualización.
     *
     * @return array|string|Response
     */
    public function actionActualizar() {
        $idRolUsuario = trim(Yii::$app->request->get("id"));
        $model = RolUsuario::findOne($idRolUsuario);

        if($model == null)
            return $this->redirect(Yii::getAlias('@web') . '/rol-usuario/');

        $listaPermisos = ArrayHelper::map(Permiso::find()->all(), "idPermiso", "permiso");

        foreach ($model->rolUsuarioPermisos as $elemento) {
            $model->permisos[] = $elemento->idPermiso;
        }

        if($model->load(Yii::$app->request->post()) && Yii::$app->request->isAjax) {
            Yii::$app->response->format = Response::FORMAT_JSON;
            return ActiveForm::validate($model);
        }

        if($model->load(Yii::$app->request->post())) {
            $idUsuarioRecord = Yii::$app->user->identity->idUsuario;
            $idRolUsuarioRecord = Yii::$app->user->identity->rolUsuario->idRolUsuario;

            try {
                if(RolUsuario::verificarExistencia($model->rolUsuario, $model->idRolUsuario))
                    throw new \Exception("Ya existe un rol de usuario registrado con ese nombre");

                $transaccion = Yii::$app->db->beginTransaction();
                $model->actualizar($idUsuarioRecord, $idRolUsuarioRecord);
                $modelRolUsuarioPermiso = new RolUsuarioPermiso();
                $modelRolUsuarioPermiso->eliminarXRolUsuario($idRolUsuario, $idUsuarioRecord, $idRolUsuarioRecord);
                $model->registrarPermisos($model->permisos, $idUsuarioRecord,
                    $idRolUsuarioRecord, $this->controlador, $this->logu, $this->logs);
                $transaccion->commit();

                $mensaje =  "SE ACTUALIZÓ EL ROL DE USUARIO CON ID: " . $model->idRolUsuario;
                $this->logu->registrar($idUsuarioRecord, $idRolUsuarioRecord, $mensaje, 1);
                $this->logs->registrar($idUsuarioRecord, $idRolUsuarioRecord, "DEBUG", $this->controlador, $mensaje);

                Yii::$app->session->setFlash('success', "Se ha actualizado el rol de usuario");
            } catch(\Exception $ex) {
                if(isset($transaccion))
                    $transaccion->rollBack();

                $mensaje =  "NO SE ACTUALIZÓ EL ROL DE USUARIO CON ID: " . $model->idRolUsuario;
                $this->logu->registrar($idUsuarioRecord, $idRolUsuarioRecord, $mensaje, 0);
                $this->logs->registrar($idUsuarioRecord, $idRolUsuarioRecord, "ERROR", $this->controlador, $ex->getMessage());

                Yii::$app->session->setFlash('error', "Error al actualizar el rol de usuario:<br />" . $ex->getMessage());
            }
        }

        return $this->render('actualizar', [
            'model' => $model,
            'listaPermisos' => $listaPermisos,
        ]);
    }

    /**
     * Muestra la pantalla de ver de un rol de usuario.
     *
     * @return string|Response
     */
    public function actionVer() {
        $idRolUsuario = trim(Yii::$app->request->get("id"));
        $model = RolUsuario::findOne($idRolUsuario);

        if($model == null)
            return $this->redirect(Yii::getAlias('@web') . '/rol-usuario/');

        $listaPermisos = ArrayHelper::map(Permiso::find()->all(), "idPermiso", "permiso");

        foreach ($model->rolUsuarioPermisos as $elemento) {
            $model->permisos[] = $elemento->idPermiso;
        }

        return $this->render('ver', [
            'model' => $model,
            'listaPermisos' => $listaPermisos,
        ]);
    }

    /**
     * Método que elimina un rol de usuario.
     *
     * @return Response
     */
    public function actionEliminar() {
        $idRolUsuario = trim(Yii::$app->request->get("id"));
        $model = RolUsuario::findOne($idRolUsuario);

        if($model == null)
            return $this->redirect(Yii::getAlias('@web') . '/rol-usuario/');

        try {
            $idUsuarioDelete = Yii::$app->user->identity->idUsuario;
            $idRolUsuarioDelete = Yii::$app->user->identity->rolUsuario->idRolUsuario;

            $transaccion = Yii::$app->db->beginTransaction();
            $modelRolUsuarioPermiso = new RolUsuarioPermiso();
            $modelRolUsuarioPermiso->eliminarXRolUsuario($idRolUsuario, $idUsuarioDelete, $idRolUsuarioDelete);
            $model->eliminar($idUsuarioDelete, $idRolUsuarioDelete);
            $transaccion->commit();

            $mensaje =  "SE ELIMINÓ EL ROL DE USUARIO CON ID: " . $idRolUsuario;
            $this->logu->registrar($idUsuarioDelete, $idRolUsuarioDelete, $mensaje, 1);
            $this->logs->registrar($idUsuarioDelete, $idRolUsuarioDelete, "DEBUG", $this->controlador, $mensaje);

            Yii::$app->session->setFlash('success', "Se ha eliminado el rol de usuario");
        } catch(IntegrityException $ex) {
            if(isset($transaccion))
                $transaccion->rollBack();

            $mensaje =  "NO SE ELIMINÓ EL ROL DE USUARIO CON ID: " . $idRolUsuario;
            $this->logu->registrar($idUsuarioDelete, $idRolUsuarioDelete, $mensaje, 0);
            $this->logs->registrar($idUsuarioDelete, $idRolUsuarioDelete, "ERROR", $this->controlador, $ex->getMessage());

            Yii::$app->session->setFlash('error', "Error al eliminar el rol de usuario:<br />" .
                "Existen datos relacionados al rol de usuario: " . $model->rolUsuario);
        } catch(\Exception $ex) {
            if(isset($transaccion))
                $transaccion->rollBack();

            $mensaje =  "NO SE ELIMINÓ EL PERFIL CON ID: " . $idRolUsuario;
            $this->logu->registrar($idUsuarioDelete, $idRolUsuarioDelete, $mensaje, 0);
            $this->logs->registrar($idUsuarioDelete, $idRolUsuarioDelete, "ERROR", $this->controlador, $ex->getMessage());

            Yii::$app->session->setFlash('error', "Error al eliminar el rol de usuario:<br />" . $ex->getMessage());
        }

        return $this->redirect(Yii::getAlias('@web') . '/rol-usuario/');
    }
}


?>