<?php
/*
 * SesionController.php
 * Springfield
 * Copyright ©: 2022 Integrasoft. Todos los derechos reservados.
 * Desarrollador(es):
 *      L.I. Humberto Javier Flores Tirado.
 *      I.S.C Julio Alberto Ocampo Rangel.
 * Fecha de creación: 25/03/2022.
 * Fecha de modificación: 17/05/2022.
 * Descripción:
 */

namespace app\controllers;

use Yii;
use yii\filters\AccessControl;
use yii\web\Controller;
use yii\web\Response;
use yii\filters\VerbFilter;
use yii\widgets\ActiveForm;
use PHPUnit\Runner\Exception;

use app\models\sesion\SesionForm;
use app\models\sesion\CambiarContraseniaForm;
use app\models\sesion\RecuperarContraseniaForm;
use app\models\sesion\TokenContrasenia;
use app\models\sesion\UsuarioSesion;
use app\models\usuario\Usuario;
use app\models\log\BitacoraUsuario;
use app\models\log\BitacoraSistema;
use app\models\sesion\VerificarSesionForm;
use app\models\sustentante\Sustentante;

/**
 * Clase controlador de sesiones.
 *
 * Class SesionController
 * @package app\controllers
 */
class SesionController extends Controller {
    private $controlador;
    private $logu;
    private $logs;

    /**
     * Método que restringue el acceso de las acciones de acuerdo a los roles.
     *
     * @return array
     */
    public function behaviors() {
        return [
            'access' => [
                'class' => AccessControl::className(),
                'only' => ['index', 'iniciar', 'cambiar-contrasenia', 'recuperar-contrasenia',
                    'recuperar-contrasenia-token'],
                'rules' => [
                    [
                        'actions' => ['index', 'cambiar-contrasenia'],
                        'allow' => true,
                        'roles' => ['@'],
                    ],
                    [
                        'actions' => ['iniciar', 'recuperar-contrasenia', 'recuperar-contrasenia-token'],
                        'allow' => true,
                        'roles' => ['?'],
                    ],
                ],
            ],
            'verbs' => [
                'class' => VerbFilter::className(),
                'actions' => [
                    'logout' => ['post'],
                ],
            ],
        ];
    }

    /**
     * Método que permite definir operaciones previas a que se ejecute un action.
     *
     * @param \yii\base\Action $action
     * @return bool
     * @throws \yii\web\BadRequestHttpException
     */
    public function beforeAction($action) {
        
        //PERMITE DESHABILITAR EL TOKEN DE SEGURIDAD CUANDO SE ENV�AN LOS FORMULARIOS
        if(in_array($action->id, ['json-iniciar-sesion-x-app', 'json-cambiar-contrasenia-x-app',
            'json-recuperar-contrasenia-x-app', 'json-actualizar-correo-x-app'])) {
            $this->enableCsrfValidation = false;
        }
//         if($action->id == 'error') {
//             if(isset(Yii::$app->user->identity))
//                 $this->layout = 'dashboard-color-admin';
//             else
//                 $this->layout = 'main';
//         }

        return parent::beforeAction($action);
    }

    /**
     * @inheritdoc
     */
    public function actions() {
        $this->controlador = "SesionController";
        $this->logu = new BitacoraUsuario();
        $this->logs = new BitacoraSistema();

        return [
            'error' => [
                //'class' => 'yii\web\ErrorAction'
                'class' => 'yii\web\ErrorAction',
            ],
            'captcha' => [
                'class' => 'yii\captcha\CaptchaAction',
                'fixedVerifyCode' => YII_ENV_TEST ? 'testme' : null,
            ],
            
        ];
    }

    public function actionStatus() {}

    /*
    * Muestra la pantalla de inicio de sesión.
    */
    public function actionIndex() {
        $this->layout = "dashboard";

        $idUsuarioCreate = Yii::$app->user->identity->idUsuario;
        $idRolUsuarioCreate = Yii::$app->user->identity->rolUsuario->idRolUsuario;
        $modelUsuario = Usuario::findOne($idUsuarioCreate);

        switch($idRolUsuarioCreate) {
            case 1:
                $mensaje = "MENSAJE";
                $alert = "alert-info";
                return $this->render('index', [
                    'mensaje' => $mensaje,
                    'alert' => $alert,
                    ]);
                break;

            case  2:
                $sustentante = Sustentante::find()->where(['idUsuario' => $idUsuarioCreate])->one();
                $mensaje = "";
                $tipoAlert = "";

                switch($sustentante->estatusSustentante->idEstatusSustentante) {
                    case 1:
                        $alert = "alert-info";
                        $mensaje = "Su registro está en espera de revisión";
                        break;

                    case 2:
                        $alert = "alert-warning";
                        $mensaje = "Su registro tiene las siguientes observaciones las cuales debe cambiar<hr />" .
                            $sustentante->observaciones;
                        break;

                    case 3:
                        $alert = "alert-info";
                        $mensaje = "Su registro está en espera de revisión";
                        break;

                    case 4:
                        $alert = "alert-info";
                        $mensaje = "Su registro está en espera de aceptación";
                        break;

                    case 5:
                        $alert = "alert-success";
                        $mensaje = "Su registro ha sido aceptado";
                        break;

                    case 6:
                        $alert = "alert-danger";
                        $mensaje = "Su registro ha sido rechazado";
                        break;

                    default:
                        $mensaje = "No se encuentra registrado";
                }

                return $this->render('index', [
                    'mensaje' => $mensaje,
                    'alert' => $alert,
                    'sustentante' => $sustentante,
                ]);

                break;
        }

        return $this->render('index');
    }

    /*
     * Muesstra la pantalla de inicio de sesión, además inicia la sesión de usuario.
     */
    public function actionIniciar() {
        $this->layout = "index-imagen";

        $model = new SesionForm();

        if($model->load(Yii::$app->request->post()) && Yii::$app->request->isAjax) {
            Yii::$app->response->format = Response::FORMAT_JSON;
            return ActiveForm::validate($model);
        }

        if($model->load(Yii::$app->request->post())) {
            $idUsuarioCreate = 0;
            $idRolUsuarioCreate = 2;

            try {
                $model->iniciar();
                
                $idUsuarioCreate = Yii::$app->user->identity->idUsuario;
                $idRolUsuarioCreate = Yii::$app->user->identity->rolUsuario->idRolUsuario;

                $mensaje =  "SE INICIO SESIÓN DEL USUARIO: " . $model->usuario;
                $this->logu->registrar($idUsuarioCreate, $idRolUsuarioCreate, $mensaje, 1);
                $this->logs->registrar($idUsuarioCreate, $idRolUsuarioCreate, "DEBUG", $this->controlador, $mensaje);
                
                $modelUsuario = Usuario::findOne($idUsuarioCreate);
                
                if($modelUsuario != null){
                    return $this->redirect(Yii::getAlias('@web') . '/sesion/');
                }
            } catch(\Exception $ex) {
                $mensaje =  "NO SE INICIÓ SESIÓN DEL USUARIO: " . $model->usuario;
                $this->logu->registrar($idUsuarioCreate, $idRolUsuarioCreate, $mensaje, 0);
                $this->logs->registrar($idUsuarioCreate, $idRolUsuarioCreate, "ERROR", $this->controlador, $ex->getMessage());

                Yii::$app->session->setFlash('error', "Error al iniciar sesión:<br />" . $ex->getMessage());
            }
        }

        return $this->render('iniciar', ['model' => $model]);
    }

    /**
     * Método que cierrar la sesión de usuario.
     *
     * @return string
     */
    public function actionCerrar() {
        $model = new SesionForm();
        try {
            $idUsuarioCreate = Yii::$app->user->identity->idUsuario;
            $idRolUsuarioCreate = Yii::$app->user->identity->rolUsuario->idRolUsuario;
            $nombreUsuario = Yii::$app->user->identity->usuario;

        
            $model->cerrar();

            $mensaje =  "SE CERRÓ SESIÓN DEL USUARIO: " . $nombreUsuario;
            $this->logu->registrar($idUsuarioCreate, $idRolUsuarioCreate, $mensaje, 1);
            $this->logs->registrar($idUsuarioCreate, $idRolUsuarioCreate, "DEBUG", $this->controlador, $mensaje);

            //Yii::$app->session->setFlash('success', "Cerrando sesi&oacute;n...");
            return $this->redirect(Yii::getAlias('@web') .'/sesion/');
            //Yii::$app->view->registerJs(
            //    "setTimeout(function(){ window.location.href ='/sesion/iniciar' }, 200);"
            //);
        } catch(\Exception $ex) {
            try {
                $mensaje =  "NO SE CERRÓ SESIÓN DEL USUARIO: " . $nombreUsuario;
                $this->logu->registrar($idUsuarioCreate, $idRolUsuarioCreate, $mensaje, 0);
                $this->logs->registrar($idUsuarioCreate, $idRolUsuarioCreate, "ERROR", $this->controlador, $ex->getMessage());
                Yii::$app->session->setFlash('error', "Error al cerrar sesión:<br />" . $ex->getMessage());
            } catch(\Exception $ex) {}
        }

        return $this->render('cerrar');
    }

    /**
     * Muesstra la pantalla de cambio de contraseña, además cambiar la contraseña del usuario.
     *
     * @return array|string
     */
    public function actionCambiarContrasenia() {
        $this->layout = "dashboard";
        $model = new CambiarContraseniaForm();

        if($model->load(Yii::$app->request->post()) && Yii::$app->request->isAjax) {
            Yii::$app->response->format = Response::FORMAT_JSON;
            return ActiveForm::validate($model);
        }

        if($model->load(Yii::$app->request->post())) {
            $idUsuarioRecord = Yii::$app->user->identity->idUsuario;
            $idRolUsuarioRecord = Yii::$app->user->identity->rolUsuario->idRolUsuario;

            try {
                $usuarioSesion = Yii::$app->user->identity;
                $usuarioSesion->cambiarContrasenia($model->contrasenia, $idUsuarioRecord, $idRolUsuarioRecord);

                $mensaje =  "SE CAMBIÓ LA CONTRASEÑA DEL USUARIO: " . $usuarioSesion->usuario;
                $this->logu->registrar($idUsuarioRecord, $idRolUsuarioRecord, $mensaje, 1);
                $this->logs->registrar($idUsuarioRecord, $idRolUsuarioRecord, "DEBUG", $this->controlador, $mensaje);

                Yii::$app->session->setFlash('success', "Se ha cambiado la contraseña");

                $model->contrasenia = null;
                $model->confirmaContrasenia = null;
            } catch(\Exception $ex) {
                $mensaje =  "NO SE CAMBIÓ LA CONTRASEÑA DEL USUARIO: " . $usuarioSesion->usuario;
                $this->logu->registrar($idUsuarioRecord, $idRolUsuarioRecord, $mensaje, 0);
                $this->logs->registrar($idUsuarioRecord, $idRolUsuarioRecord, "ERROR", $this->controlador, $ex->getMessage());

                Yii::$app->session->setFlash('error', "Error al actualizar la contraseña:<br />" . $ex->getMessage());
            }
        }

        return $this->render('cambiar-contrasenia', ['model' => $model]);
    }

    /**
     * Muestra la pantalla para recuperar la contraseña.
     *
     * @return array|string
     */
    public function actionRecuperarContrasenia() {
        $model = new RecuperarContraseniaForm();

        if($model->load(Yii::$app->request->post()) && Yii::$app->request->isAjax) {
            Yii::$app->response->format = Response::FORMAT_JSON;
            return ActiveForm::validate($model);
        }

        if($model->load(Yii::$app->request->post())) {
            $idUsuarioCreate = 0;
            $idRolUsuarioCreate = 3;
            $nombreUsuario = "";

            try {
                $usuarioSesion = UsuarioSesion::getUsuarioSesion($model->usuario, null, null);
                
                if($usuarioSesion == null)
                    throw new \Exception("No existe un usuario registrado");

                $nombreUsuario = $usuarioSesion->usuario;

                if(UsuarioSesion::recuperarContrasenia($model->usuario)) {
                    $idUsuarioCreate = $usuarioSesion->idUsuario;
                    $idRolUsuarioCreate = $usuarioSesion->rolUsuario->idRolUsuario;

                    $mensaje =  "SE GENERÓ EL TOKEN DEL USUARIO: " . $nombreUsuario;
                    $this->logu->registrar($idUsuarioCreate, $idRolUsuarioCreate, $mensaje, 1);
                    $this->logs->registrar($idUsuarioCreate, $idRolUsuarioCreate, "DEBUG", $this->controlador, $mensaje);

                    Yii::$app->session->setFlash('success', "Se ha enviado un correo electrónico para recuperar la contraseña");

                    $model->usuario = null;
                }
                else
                    throw new \Exception("Error al generar el token para recuperar la contraseña");

            } catch(\Exception $ex) {
                $mensaje =  "NO SE GENERÓ EL TOKEN DEL USUARIO: " . $nombreUsuario;
                $this->logu->registrar($idUsuarioCreate, $idRolUsuarioCreate, $mensaje, 0);
                $this->logs->registrar($idUsuarioCreate, $idRolUsuarioCreate, "ERROR", $this->controlador, $ex->getMessage());

                Yii::$app->session->setFlash('error', "Error al recuperar la contraseña:<br />" . $ex->getMessage());
            }
        }

        return $this->render('recuperar-contrasenia', ['model' => $model]);
    }
    
    /**
     * Muetra la pantalla para recuperar la contraseña ingresando la nueva contraseña, además de cambiar dicha contraseña.
     *
     * @return array|string
     * @throws \yii\db\Exception
     */
    public function actionRecuperarContraseniaToken() {
        $model = new CambiarContraseniaForm();
        $token = trim(Yii::$app->request->get("token"));

        if($model->load(Yii::$app->request->post()) && Yii::$app->request->isAjax) {
            Yii::$app->response->format = Response::FORMAT_JSON;
            return ActiveForm::validate($model);
        }

        if($model->load(Yii::$app->request->post())) {
            $idUsuarioRecord = 0;
            $idRolUsuarioRecord = 0;

            try {
                $tokenContrasenia = TokenContrasenia::getXToken($token);

                if($tokenContrasenia == null)
                    throw new \Exception("No existe un token registrado con ese enlace");

                $usuarioSesion = UsuarioSesion::getUsuarioSesion(null, null, $tokenContrasenia->idSesion);

                if($usuarioSesion == null)
                    throw new \Exception("No existe un usuario relacionado el enlace/token ingresado");

                $idUsuarioRecord = $usuarioSesion->idUsuario;
                $idRolUsuarioRecord = $usuarioSesion->rolUsuario->idRolUsuario;

                $transaccion = Yii::$app->db->beginTransaction();
                $tokenContrasenia->eliminar($idUsuarioRecord, $idRolUsuarioRecord);
                $usuarioSesion->cambiarContrasenia($model->contrasenia, $idUsuarioRecord, $idRolUsuarioRecord);
                $transaccion->commit();

                $mensaje =  "SE CAMBIÓ LA CONTRASEÑA DEL USUARIO: " . $usuarioSesion->usuario;
                $this->logu->registrar($idUsuarioRecord, $idRolUsuarioRecord, $mensaje, 1);
                $this->logs->registrar($idUsuarioRecord, $idRolUsuarioRecord, "DEBUG", $this->controlador, $mensaje);

                Yii::$app->session->setFlash('success', "Se ha cambiado la contraseña");

                $model->contrasenia = null;
                $model->confirmaContrasenia = null;
            } catch(\Exception $ex) {
                if(isset($transaccion))
                    $transaccion->rollBack();

                $mensaje =  "NO SE CAMBIÓ LA CONTRASEÑA DEL TOKEN : " . $token;
                $this->logu->registrar($idUsuarioRecord, $idRolUsuarioRecord, $mensaje, 0);
                $this->logs->registrar($idUsuarioRecord, $idRolUsuarioRecord, "ERROR", $this->controlador, $ex->getMessage());

                Yii::$app->session->setFlash('error', "Error al recuperar la contraseña:<br />" . $ex->getMessage());
            }

        }

        return $this->render('recuperar-contrasenia-token', ['model' => $model]);
    }

}

?>