<?php
/*
 * SustentanteController.php
 * Springfield
 * Copyright ©: 2022 Integrasoft. Todos los derechos reservados.
 * Desarrollador(es):
 *      I.S.C. Julio Alberto Ocampo Rangel.
 * Fecha de creación: 28/03/2022.
 * Fecha de modificación: 20/05/2022.
 * Descripción:
 */

namespace app\controllers;

use Yii;
use yii\filters\AccessControl;
use yii\filters\VerbFilter;
use yii\helpers\ArrayHelper;
use yii\helpers\Url;
use yii\web\Controller;
use yii\web\Response;
use yii\widgets\ActiveForm;
use yii\db\IntegrityException;
use yii\helpers\Json;
use yii\web\UploadedFile;

use app\models\log\BitacoraUsuario;
use app\models\log\BitacoraSistema;

use app\models\areaconocimiento\AreaConocimiento;
use app\models\sede\Sede;
use app\models\sexo\Sexo;
use app\models\estado\Estado;
use app\models\perfilprofesional\PerfilProfesional;
use app\models\motivonoconcluyo\MotivoNoConcluyo;
use app\models\niveleducativo\NivelEducativo;
use app\models\antecedenteacademico\AntecedenteAcademico;
use app\models\estatussustentante\EstatusSustentante;
use app\models\rolusuario\SustentanteSearch as RolusuarioSustentanteSearch;
use app\models\sustentante\Sustentante;
use app\models\sustentante\SustentanteSearch;
use app\models\usuario\Usuario;
use app\models\sesion\UsuarioSesion;
use app\models\util\Utils;

/**
 * Clase controlador de sustentantes.
 *
 * Class SustentanteController
 * @package app\controllers
 */
class SustentanteController extends Controller {
    private $controlador;
    private $logu;
    private $logs;

    /**
     * Método que restringue el acceso de las acciones de acuerdo a los roles.
     *
     * @return array
     */
    public function behaviors() {
        return [
            'access' => [
                'class' => AccessControl::class,
                'only' => ['index', 'registrar', 'actualizar', 'validar', 'ver', 'eliminar', 'exportar'],
                'rules' => [
                    [
                        'actions' => ['index', 'registrar', 'actualizar', 'validar', 'ver', 'eliminar', 'exportar'],
                        'allow' => true,
                        'roles' => ['@'],
                        'matchCallback' => function($rule, $action) {
                            $permiso = null;

                            switch($action->id) {
                                case 'index':
                                    $permiso = 'Buscar sustentante';
                                    break;

                                case 'registrar':
                                    $permiso = 'Registrar sustentante';
                                    break;

                                case 'actualizar':
                                    $permiso = 'Actualizar sustentante';
                                    break;

                                case 'validar':
                                    $permiso = 'Actualizar sustentante';
                                    break;

                                case 'ver':
                                    $permiso = 'Ver sustentante';
                                    break;

                                case 'eliminar':
                                    $permiso = 'Eliminar sustentante';
                                    break;

                                case 'exportar':
                                    $permiso = 'Exportar sustentantes';
                                    break;

                                default:
                                    return false;
                            }

                            if(isset(Yii::$app->user->identity))
                                return Yii::$app->user->identity->datos->verificarPermisos($permiso);
                            else
                                return false;
                        }
                    ],
                    [
                        'actions' => ['registrarse'],
                        'allow' => true,
                        'roles' => ['?'],
                    ],
                ],
            ],
            'verbs' => [
                'class' => VerbFilter::class,
                'actions' => [
                    'logout' => ['post'],
                ],
            ],
        ];
    }

    /**
     * @inheritdoc
     */
    public function actions() {
        //$this->layout = "dashboard";
        $this->controlador = "RolUsuarioController";
        $this->logu = new BitacoraUsuario();
        $this->logs = new BitacoraSistema();

        return [
            'error' => [
                'class' => 'yii\web\ErrorAction',
            ],
            'captcha' => [
                'class' => 'yii\captcha\CaptchaAction',
                'fixedVerifyCode' => YII_ENV_TEST ? 'testme' : null,
            ],

        ];
    }

    public function actionStatus() {}

    /**
     * Muestra la pantalla de gestión de roles de usuario.
     */
    public function actionIndex() {
        $this->layout = "dashboard";
        $searchModel = new SustentanteSearch();
        $dataProvider = $searchModel->search(Yii::$app->request->queryParams);

        if(Yii::$app->user->identity->idRolUsuario == 3) {
            $searchModel->idSede = Yii::$app->user->identity->datos->sede->idSede;
            $listaSede = ArrayHelper::map(Sede::find()->where(['idSede' => Yii::$app->user->identity->datos->sede->idSede])->all(), "idSede", "sede");
        } else {
            $listaSede = ArrayHelper::map(Sede::find()->all(), "idSede", "sede");
        }

        $listaSexo = ArrayHelper::map(Sexo::find()->all(), "idSexo", "sexo");
        $listaEstado = ArrayHelper::map(Estado::find()->all(), "idEstado", "estado");
        $listaNivelEducativo = ArrayHelper::map(NivelEducativo::find()->all(), "idNivelEducativo", "nivelEducativo");
        $listaAreaConocimiento = ArrayHelper::map(AreaConocimiento::find()->all(), "idAreaConocimiento", "areaConocimiento");
        $listaPerfilProfesional = ArrayHelper::map(PerfilProfesional::find()->all(), "idPerfilProfesional", "perfilProfesional");
        $listaMotivoNoConcluyo = ArrayHelper::map(MotivoNoConcluyo::find()->all(), "idMotivoNoConcluyo", "motivoNoConcluyo");
        $listaAntecedenteAcademico = ArrayHelper::map(AntecedenteAcademico::find()->all(), "idAntecedenteAcademico", "antecedenteAcademico");
        $listaEstatusSustentante = ArrayHelper::map(EstatusSustentante::find()->all(), "idEstatusSustentante", "estatusSustentante");

        return $this->render('index', [
            'searchModel' => $searchModel,
            'dataProvider' => $dataProvider,
            'listaSede' => $listaSede,
            'listaSexo' => $listaSexo,
            'listaEstado' => $listaEstado,
            'listaNivelEducativo' => $listaNivelEducativo,
            'listaAreaConocimiento' => $listaAreaConocimiento,
            'listaPerfilProfesional' => $listaPerfilProfesional,
            'listaMotivoNoConcluyo' => $listaMotivoNoConcluyo,
            'listaAntecedenteAcademico' => $listaAntecedenteAcademico,
            'listaEstatusSustentante' => $listaEstatusSustentante
        ]);
    }

    /**
     * Muestra la pantalla de registro de un rol de usuario, además realiza el registro.
     *
     * @return array|string
     */
    public function actionRegistrar() {
        $this->layout = "dashboard";
        $model = new Sustentante();
        $model->scenario = "registrar";
        $modelUsuario = new Usuario();

        $idUsuarioCreate = Yii::$app->user->identity->idUsuario;
        $idRolUsuarioCreate = Yii::$app->user->identity->rolUsuario->idRolUsuario;

        $listaAreaConocimiento = ArrayHelper::map(AreaConocimiento::find()->all(), "idAreaConocimiento", "areaConocimiento");
        if($idRolUsuarioCreate != 3) {
            $listaSede = ArrayHelper::map(Sede::find()->all(), "idSede", "sede");
        }else {
            $listaSede = ArrayHelper::map(Sede::find()->where(['idSede' => Yii::$app->user->identity->datos->sede->idSede])->all(), "idSede", "sede");
        }
        $listaSexo = ArrayHelper::map(Sexo::find()->all(), "idSexo", "sexo");
        $listaEstado = ArrayHelper::map(Estado::find()->all(), "idEstado", "estado");
        $listaNivelEducativo = ArrayHelper::map(NivelEducativo::find()->all(), "idNivelEducativo", "nivelEducativo");
        $listaMotivoNoConcluyo = ArrayHelper::map(MotivoNoConcluyo::find()->all(), "idMotivoNoConcluyo", "motivoNoConcluyo");
        $listaAntecedenteAcademico = ArrayHelper::map(AntecedenteAcademico::find()->all(), "idAntecedenteAcademico", "antecedenteAcademico");

        if($model->load(Yii::$app->request->post()) && Yii::$app->request->isAjax) {
            Yii::$app->response->format = Response::FORMAT_JSON;
            return ActiveForm::validate($model);
        }

        if($model->load(Yii::$app->request->post())) {
            

            try {
                if(Sustentante::verificarExistenciaCurp($model->curp))
                    throw new \Exception("Ya existe un sustentante registrado con esa CURP");

                if(Sustentante::verificarFechaNacimientoMenorAFechaActual($model->fechaNacimiento))
                    throw new \Exception("Fecha de nacimiento debe ser menor a la fecha actual");

                if(Usuario::verificarExistencia($model->correoElectronico))
                    throw new \Exception("Ya existe un sustentante registrado con ese correo");

                if(!UsuarioSesion::verificarContraseniaValida($model->contrasenia))
                    throw new \Exception("La contraseña debe contener por lo menos dos letras, un número y un caracter especial.");

                $transaccion = Yii::$app->db->beginTransaction();
                $modelUsuario->idRolUsuario = 2;
                $modelUsuario->idEstado = $model->idEstadoRadica;
                $modelUsuario->idEstatusUsuario = 1;
                $modelUsuario->usuario = $model->correoElectronico;
                $modelUsuario->contrasenia = $model->contrasenia;
                $modelUsuario->idSede = $model->idSede;
                $modelUsuario->nombre = $model->nombre;
                $modelUsuario->primerApellido = $model->primerApellido;
                $modelUsuario->segundoApellido = $model->segundoApellido;
                $modelUsuario->registrar($idUsuarioCreate, $idRolUsuarioCreate);
                $model->idUsuario = $modelUsuario->idUsuario;
                $model->archivoActaNacimiento = '';
                $model->archivoCurp = '';
                $model->archivoIne = '';
                $model->archivoCertificado = '';
                $model->archivoPersonaAutorizada1Upload = UploadedFile::getInstance($model, 'archivoPersonaAutorizada1Upload');
                $model->archivoPersonaAutorizada2Upload = UploadedFile::getInstance($model, 'archivoPersonaAutorizada2Upload');
                $model->archivoActaNacimientoUpload = UploadedFile::getInstance($model, 'archivoActaNacimientoUpload');
                $model->archivoCurpUpload = UploadedFile::getInstance($model, 'archivoCurpUpload');
                $model->archivoIneUpload = UploadedFile::getInstance($model, 'archivoIneUpload');
                $model->archivoCertificadoUpload = UploadedFile::getInstance($model, 'archivoCertificadoUpload');
                $model->archivoAntecedenteSecundariaUpload = UploadedFile::getInstance($model, 'archivoAntecedenteSecundariaUpload');
                $model->archivoCurriculumUpload = UploadedFile::getInstance($model, 'archivoCurriculumUpload');
                $model->archivoDoctoUltimoGradoEstudiosUpload = UploadedFile::getInstance($model, 'archivoDoctoUltimoGradoEstudiosUpload');
                $model->archivoCartaAutorizacionUpload = UploadedFile::getInstance($model, 'archivoCartaAutorizacionUpload');
                /*
                 *  ESTATUS VALIDADO SI VIENE DEL ROL ADMINISTRATIVO
                 *  ESTATUS REGISTRADO POR VALIDAR SI BIENE DEL ROL SUSTENTANTE
                 */

                if($idRolUsuarioCreate == 1)
                    $model->idEstatusSustentante = 5;
                elseif($idRolUsuarioCreate == 3)
                    $model->idEstatusSustentante = 4;

                $model->registrar($idUsuarioCreate, $idRolUsuarioCreate);
                $model->mesNumeroControl = Sustentante::obtenerLetasDelMes(date("m"));
                $model->anioNumeroControl = date("y");
                $model->consecutivoNumeroControl = Sustentante::generarConsecutivo($model->anioNumeroControl);
                $model->numeroControl = $model->anioNumeroControl . $model->mesNumeroControl . Sustentante::formatearConsecutivo($model->consecutivoNumeroControl);
                $model->actualizar($idUsuarioCreate, $idRolUsuarioCreate);
                $transaccion->commit();

                $mensaje =  "SE REGISTRÓ EL SUSTENTANTE CON ID: " . $model->idSustentante;
                $this->logu->registrar($idUsuarioCreate, $idRolUsuarioCreate, $mensaje, 1);
                $this->logs->registrar($idUsuarioCreate, $idRolUsuarioCreate, "DEBUG", $this->controlador, $mensaje);

                Yii::$app->session->setFlash('success', "Se ha registrado el sustentante");

                $model = new Sustentante();
            } catch(\Exception $ex) {
                if(isset($transaccion))
                    $transaccion->rollBack();

                $mensaje =  "NO SE REGISTRÓ EL SUSTENTANTE CON NOMBRE: " . $model->nombre;
                $this->logu->registrar($idUsuarioCreate, $idRolUsuarioCreate, $mensaje, 0);
                $this->logs->registrar($idUsuarioCreate, $idRolUsuarioCreate, "ERROR", $this->controlador, $ex->getMessage());

                Yii::$app->session->setFlash('error', "Error al registrar al sustentante:<br />" . $ex->getMessage());
            }
        }

        return $this->render('registrar', [
            'model' => $model,
            'listaAreaConocimiento' => $listaAreaConocimiento,
            'listaSede' => $listaSede,
            'listaSexo' => $listaSexo,
            'listaEstado' => $listaEstado,
            'listaNivelEducativo' => $listaNivelEducativo,
            'listaMotivoNoConcluyo' => $listaMotivoNoConcluyo,
            'listaAntecedenteAcademico' => $listaAntecedenteAcademico
        ]);
    }
    
    /**
     * Muestra la pantalla para recuperar la contraseña.
     *
     * @return array|string
     */
    public function actionRegistrarse() {
        //$this->layout = "index";
        $model = new Sustentante();
        $model->scenario = "registrar";
        $modelUsuario = new Usuario();

        $listaAreaConocimiento = ArrayHelper::map(AreaConocimiento::find()->all(), "idAreaConocimiento", "areaConocimiento");
        $listaSede = ArrayHelper::map(Sede::find()->all(), "idSede", "sede");
        $listaSexo = ArrayHelper::map(Sexo::find()->all(), "idSexo", "sexo");
        $listaEstado = ArrayHelper::map(Estado::find()->all(), "idEstado", "estado");
        $listaNivelEducativo = ArrayHelper::map(NivelEducativo::find()->all(), "idNivelEducativo", "nivelEducativo");
        $listaMotivoNoConcluyo = ArrayHelper::map(MotivoNoConcluyo::find()->all(), "idMotivoNoConcluyo", "motivoNoConcluyo");
        $listaAntecedenteAcademico = ArrayHelper::map(AntecedenteAcademico::find()->all(), "idAntecedenteAcademico", "antecedenteAcademico");

        if($model->load(Yii::$app->request->post()) && Yii::$app->request->isAjax) {
            Yii::$app->response->format = Response::FORMAT_JSON;
            return ActiveForm::validate($model);
        }

        if($model->load(Yii::$app->request->post())) {
            $idUsuarioCreate = 0;
            $idRolUsuarioCreate = 3;

            try {
                if(Sustentante::verificarExistenciaCurp($model->curp))
                    throw new \Exception("Ya existe un sustentante registrado con esa CURP");

                if(Sustentante::verificarFechaNacimientoMenorAFechaActual($model->fechaNacimiento))
                    throw new \Exception("Fecha de nacimiento debe ser menor a la fecha actual");

                if(Usuario::verificarExistencia($model->correoElectronico))
                    throw new \Exception("Ya existe un sustentante registrado con ese correo");

                if(!UsuarioSesion::verificarContraseniaValida($model->contrasenia))
                    throw new \Exception("La contraseña debe contener por lo menos dos letras, un número y un caracter especial.");

                $transaccion = Yii::$app->db->beginTransaction();
                $modelUsuario->idRolUsuario = 2;
                $modelUsuario->idEstado = $model->idEstadoRadica;
                $modelUsuario->idEstatusUsuario = 1;
                $modelUsuario->usuario = $model->correoElectronico;
                $modelUsuario->contrasenia = $model->contrasenia;
                $modelUsuario->idSede = $model->idSede;
                $modelUsuario->nombre = $model->nombre;
                $modelUsuario->primerApellido = $model->primerApellido;
                $modelUsuario->segundoApellido = $model->segundoApellido;
                $modelUsuario->registrar($idUsuarioCreate, $idRolUsuarioCreate);
                $model->idUsuario = $modelUsuario->idUsuario;
                $model->archivoActaNacimiento = '';
                $model->archivoCurp = '';
                $model->archivoIne = '';
                $model->archivoCertificado = '';
                $model->archivoPersonaAutorizada1Upload = UploadedFile::getInstance($model, 'archivoPersonaAutorizada1Upload');
                $model->archivoPersonaAutorizada2Upload = UploadedFile::getInstance($model, 'archivoPersonaAutorizada2Upload');
                $model->archivoActaNacimientoUpload = UploadedFile::getInstance($model, 'archivoActaNacimientoUpload');
                $model->archivoCurpUpload = UploadedFile::getInstance($model, 'archivoCurpUpload');
                $model->archivoIneUpload = UploadedFile::getInstance($model, 'archivoIneUpload');
                $model->archivoCertificadoUpload = UploadedFile::getInstance($model, 'archivoCertificadoUpload');
                $model->archivoAntecedenteSecundariaUpload = UploadedFile::getInstance($model, 'archivoAntecedenteSecundariaUpload');
                $model->archivoCurriculumUpload = UploadedFile::getInstance($model, 'archivoCurriculumUpload');
                $model->archivoDoctoUltimoGradoEstudiosUpload = UploadedFile::getInstance($model, 'archivoDoctoUltimoGradoEstudiosUpload');
                $model->archivoCartaAutorizacionUpload = UploadedFile::getInstance($model, 'archivoCartaAutorizacionUpload');
                /*
                 *  ESTATUS VALIDADO SI VIENE DEL ROL ADMINISTRATIVO
                 *  ESTATUS REGISTRADO POR VALIDAR SI BIENE DEL ROL SUSTENTANTE
                 */

                $model->idEstatusSustentante = 1;
                $model->registrar($idUsuarioCreate, $idRolUsuarioCreate);
                $transaccion->commit();

                $mensaje =  "SE REGISTRÓ EL SUSTENTANTE CON ID: " . $model->idSustentante;
                $this->logu->registrar($idUsuarioCreate, $idRolUsuarioCreate, $mensaje, 1);
                $this->logs->registrar($idUsuarioCreate, $idRolUsuarioCreate, "DEBUG", $this->controlador, $mensaje);

                Yii::$app->session->setFlash('success', "Se ha registrado el sustentante");
                return $this->redirect(Yii::getAlias('@web') . '/');
            } catch(\Exception $ex) {
                $mensaje =  "NO SE REGISTRÓ EL SUSTENTANTE";
                $this->logu->registrar($idUsuarioCreate, $idRolUsuarioCreate, $mensaje, 0);
                $this->logs->registrar($idUsuarioCreate, $idRolUsuarioCreate, "ERROR", $this->controlador, $ex->getMessage());

                Yii::$app->session->setFlash('error', "Error:<br />" . $ex->getMessage());
            }
        }

        return $this->render('registrarse', [
            'model' => $model,
            'listaAreaConocimiento' => $listaAreaConocimiento,
            'listaSede' => $listaSede,
            'listaSexo' => $listaSexo,
            'listaEstado' => $listaEstado,
            'listaNivelEducativo' => $listaNivelEducativo,
            'listaMotivoNoConcluyo' => $listaMotivoNoConcluyo,
            'listaAntecedenteAcademico' => $listaAntecedenteAcademico
        ]);
    }

    /**
     * Muestra la pantalla de actualización de un rol de usuario, además realiza la actualización.
     *
     * @return array|string|Response
     */
    public function actionActualizar() {
        $this->layout = "dashboard";
        $idSustentante = 0;
        $idUsuarioRecord = Yii::$app->user->identity->idUsuario;
        $idRolUsuarioRecord = Yii::$app->user->identity->rolUsuario->idRolUsuario;
        $model = null;
        if(Yii::$app->user->identity->rolUsuario->idRolUsuario == 1 ||
            Yii::$app->user->identity->rolUsuario->idRolUsuario == 3) {
            $idSustentante = trim(Yii::$app->request->get("id"));
            $model = Sustentante::findOne($idSustentante);
        } else {
            $model = Sustentante::find()->where(['idUsuario' => Yii::$app->user->identity->idUsuario])->one();
            if($model->idEstatusSustentante != 2){
                Yii::$app->session->setFlash('error', "No se puede actualizar el registro");
                return $this->redirect(Yii::getAlias('@web') . '/');
            }
            $model->idEstatusSustentante = 3;
        }
        $model->scenario = "actualizar";

        if($model == null)
            return $this->redirect(Yii::getAlias('@web') . '/sustentante/');

        if(Yii::$app->user->identity->rolUsuario->idRolUsuario == 1) {
            if($model->idEstatusSustentante == 5 ||
                $model->idEstatusSustentante == 6) {
                Yii::$app->session->setFlash('error', "No se puede actualizar un sustentante con estatus: Aceptado o Rechazado");
                return $this->redirect(Yii::getAlias('@web') . '/sustentante/');
            }
        } elseif(Yii::$app->user->identity->rolUsuario->idRolUsuario == 3) {
            if($model->idEstatusSustentante == 5 ||
                $model->idEstatusSustentante == 6) {
                Yii::$app->session->setFlash('error', "No se puede actualizar un sustentante con estatus: Revisado, Aceptado o Rechazado");
                return $this->redirect(Yii::getAlias('@web') . '/sustentante/');
            }
        }

        $listaAreaConocimiento = ArrayHelper::map(AreaConocimiento::find()->all(), "idAreaConocimiento", "areaConocimiento");
        $modelPerfilProfesional = PerfilProfesional::findOne($model->idPerfilProfesional);
        $listaPerfilProfesional = [];
        if($modelPerfilProfesional != null) {
            $listaPerfilProfesional = ArrayHelper::map(PerfilProfesional::find()->where(['idAreaConocimiento' => $modelPerfilProfesional->idAreaConocimiento])->all(), "idPerfilProfesional", "perfilProfesional");
            $model->idAreaConocimiento = $modelPerfilProfesional->idAreaConocimiento;
        }

        if($idRolUsuarioRecord != 3) {
            $listaSede = ArrayHelper::map(Sede::find()->all(), "idSede", "sede");
        }else {
            $listaSede = ArrayHelper::map(Sede::find()->where(['idSede' => Yii::$app->user->identity->datos->sede->idSede])->all(), "idSede", "sede");
        }
        $listaSexo = ArrayHelper::map(Sexo::find()->all(), "idSexo", "sexo");
        $listaEstado = ArrayHelper::map(Estado::find()->all(), "idEstado", "estado");
        $listaNivelEducativo = ArrayHelper::map(NivelEducativo::find()->all(), "idNivelEducativo", "nivelEducativo");
        $listaMotivoNoConcluyo = ArrayHelper::map(MotivoNoConcluyo::find()->all(), "idMotivoNoConcluyo", "motivoNoConcluyo");
        $listaAntecedenteAcademico = ArrayHelper::map(AntecedenteAcademico::find()->all(), "idAntecedenteAcademico", "antecedenteAcademico");

        $model->idSede = $model->usuario->idSede;
        $model->nombre = $model->usuario->nombre;
        $model->primerApellido = $model->usuario->primerApellido;
        $model->segundoApellido = $model->usuario->segundoApellido;
        $model->correoElectronico = $model->usuario->usuario;

        $personaAutorizada1Archivo = Yii::getAlias('@web') . '/sustentante/descargar/?id=' . $idSustentante .'&tipoArchivo=' . $model->archivoPersonaAutorizada1 . '&tipo=imagen';
        $personaAutorizada2Archivo = Yii::getAlias('@web') . '/sustentante/descargar/?id=' . $idSustentante .'&tipoArchivo=' . $model->archivoPersonaAutorizada2 . '&tipo=imagen';
        $actaNacimientoArchivo = Yii::getAlias('@web') . '/sustentante/descargar/?id=' . $idSustentante .'&tipoArchivo=' . $model->archivoActaNacimiento . '&tipo=imagen';
        $curpArchivo = Yii::getAlias('@web') . '/sustentante/descargar/?id=' . $idSustentante .'&tipoArchivo=' . $model->archivoCurp . '&tipo=imagen';
        $ineArchivo = Yii::getAlias('@web') . '/sustentante/descargar/?id=' . $idSustentante .'&tipoArchivo=' . $model->archivoIne . '&tipo=imagen';
        $certificadoArchivo = Yii::getAlias('@web') . '/sustentante/descargar/?id=' . $idSustentante .'&tipoArchivo=' . $model->archivoCertificado . '&tipo=imagen';
        $antecedenteSecundariaArchivo = Yii::getAlias('@web') . '/sustentante/descargar/?id=' . $idSustentante .'&tipoArchivo=' . $model->archivoAntecedenteSecundaria . '&tipo=imagen';
        $curriculumArchivo = Yii::getAlias('@web') . '/sustentante/descargar/?id=' . $idSustentante .'&tipoArchivo=' . $model->archivoCurriculum . '&tipo=imagen';
        $doctoUltimoGradoEstudiosArchivo = Yii::getAlias('@web') . '/sustentante/descargar/?id=' . $idSustentante .'&tipoArchivo=' . $model->archivoDoctoUltimoGradoEstudios . '&tipo=imagen';
        $cartaAutorizacionArchivo = Yii::getAlias('@web') . '/sustentante/descargar/?id=' . $idSustentante .'&tipoArchivo=' . $model->archivoCartaAutorizacion . '&tipo=imagen';

        if($model->load(Yii::$app->request->post()) && Yii::$app->request->isAjax) {
            Yii::$app->response->format = Response::FORMAT_JSON;
            return ActiveForm::validate($model);
        }

        if($model->load(Yii::$app->request->post())) {
            try {
                if(Sustentante::verificarExistenciaCurp($model->curp, $model->idSustentante))
                    throw new \Exception("Ya existe un sustentante registrado con esa CURP");

                if(Sustentante::verificarFechaNacimientoMenorAFechaActual($model->fechaNacimiento))
                    throw new \Exception("Fecha de nacimiento debe ser menor a la fecha actual");

                if(Usuario::verificarExistencia($model->usuario->usuario, $model->idUsuario))
                    throw new \Exception("Ya existe un sustentante registrado con ese correo");

                $transaccion = Yii::$app->db->beginTransaction();
                $model->archivoPersonaAutorizada1Upload = UploadedFile::getInstance($model, 'archivoPersonaAutorizada1Upload');
                $model->archivoPersonaAutorizada2Upload = UploadedFile::getInstance($model, 'archivoPersonaAutorizada2Upload');
                $model->archivoActaNacimientoUpload = UploadedFile::getInstance($model, 'archivoActaNacimientoUpload');
                $model->archivoCurpUpload = UploadedFile::getInstance($model, 'archivoCurpUpload');
                $model->archivoIneUpload = UploadedFile::getInstance($model, 'archivoIneUpload');
                $model->archivoCertificadoUpload = UploadedFile::getInstance($model, 'archivoCertificadoUpload');
                $model->archivoAntecedenteSecundariaUpload = UploadedFile::getInstance($model, 'archivoAntecedenteSecundariaUpload');
                $model->archivoCurriculumUpload = UploadedFile::getInstance($model, 'archivoCurriculumUpload');
                $model->archivoDoctoUltimoGradoEstudiosUpload = UploadedFile::getInstance($model, 'archivoDoctoUltimoGradoEstudiosUpload');
                $model->archivoCartaAutorizacionUpload = UploadedFile::getInstance($model, 'archivoCartaAutorizacionUpload');
                $model->actualizar($idUsuarioRecord, $idRolUsuarioRecord);
                $transaccion->commit();

                $mensaje =  "SE ACTUALIZÓ EL SUSTENTANTE CON ID: " . $model->idSustentante;
                $this->logu->registrar($idUsuarioRecord, $idRolUsuarioRecord, $mensaje, 1);
                $this->logs->registrar($idUsuarioRecord, $idRolUsuarioRecord, "DEBUG", $this->controlador, $mensaje);

                Yii::$app->session->setFlash('success', "Se ha actualizado el sustentante");
                if(Yii::$app->user->identity->rolUsuario->idRolUsuario == 2)
                    return $this->redirect(Yii::getAlias('@web') . '/');

            } catch(\Exception $ex) {
                if(isset($transaccion))
                    $transaccion->rollBack();

                $mensaje =  "NO SE ACTUALIZÓ EL SUSTENTANTE CON ID: " . $model->idSustentante;
                $this->logu->registrar($idUsuarioRecord, $idRolUsuarioRecord, $mensaje, 0);
                $this->logs->registrar($idUsuarioRecord, $idRolUsuarioRecord, "ERROR", $this->controlador, $ex->getMessage());

                Yii::$app->session->setFlash('error', "Error al actualizar el sustentante:<br />" . $ex->getMessage());
            }
        }

        return $this->render('actualizar', [
            'model' => $model,
            'listaAreaConocimiento' => $listaAreaConocimiento,
            'listaPerfilProfesional' => $listaPerfilProfesional,
            'listaSede' => $listaSede,
            'listaSexo' => $listaSexo,
            'listaEstado' => $listaEstado,
            'listaNivelEducativo' => $listaNivelEducativo,
            'listaMotivoNoConcluyo' => $listaMotivoNoConcluyo,
            'listaAntecedenteAcademico' => $listaAntecedenteAcademico,
            'personaAutorizada1Archivo' => $personaAutorizada1Archivo,
            'personaAutorizada2Archivo' => $personaAutorizada2Archivo,
            'actaNacimientoArchivo' => $actaNacimientoArchivo,
            'curpArchivo' => $curpArchivo,
            'ineArchivo' => $ineArchivo,
            'certificadoArchivo' => $certificadoArchivo,
            'antecedenteSecundariaArchivo' => $antecedenteSecundariaArchivo,
            'curriculumArchivo' => $curriculumArchivo,
            'doctoUltimoGradoEstudiosArchivo' => $doctoUltimoGradoEstudiosArchivo,
            'cartaAutorizacionArchivo' => $cartaAutorizacionArchivo
        ]);
    }

    /**
     * Muestra la pantalla de actualización de un rol de usuario, además realiza la actualización.
     *
     * @return array|string|Response
     */
    public function actionValidar() {
        $this->layout = "dashboard";
        $idSustentante = trim(Yii::$app->request->get("id"));
        $model = Sustentante::findOne($idSustentante);

        if($model == null)
            return $this->redirect(Yii::getAlias('@web') . '/sustentante/');

        if(Yii::$app->user->identity->rolUsuario->idRolUsuario == 1) {
            if($model->idEstatusSustentante == 5) {
                Yii::$app->session->setFlash('error', "No se puede validar un sustentante con estatus: Aceptado");
                return $this->redirect(Yii::getAlias('@web') . '/sustentante/');
            }
        } elseif(Yii::$app->user->identity->rolUsuario->idRolUsuario == 3) {
            if($model->idEstatusSustentante == 4 ||
                $model->idEstatusSustentante == 5 ||
                $model->idEstatusSustentante == 6) {
                Yii::$app->session->setFlash('error', "No se puede validar un sustentante con estatus: Revisado, Aceptado o Rechazado");
                return $this->redirect(Yii::getAlias('@web') . '/sustentante/');
            }
        }

        $idUsuarioRecord = Yii::$app->user->identity->idUsuario;
        $idRolUsuarioRecord = Yii::$app->user->identity->rolUsuario->idRolUsuario;

        $listaAreaConocimiento = ArrayHelper::map(AreaConocimiento::find()->all(), "idAreaConocimiento", "areaConocimiento");
        $modelPerfilProfesional = PerfilProfesional::findOne($model->idPerfilProfesional);
        $listaPerfilProfesional = [];
        if($modelPerfilProfesional != null) {
            $listaPerfilProfesional = ArrayHelper::map(PerfilProfesional::find()->where(['idAreaConocimiento' => $modelPerfilProfesional->idAreaConocimiento])->all(), "idPerfilProfesional", "perfilProfesional");
            $model->idAreaConocimiento = $modelPerfilProfesional->idAreaConocimiento;
        }
        $model->scenario = "validar";
        if($idRolUsuarioRecord != 3) {
            $listaSede = ArrayHelper::map(Sede::find()->all(), "idSede", "sede");
        }else {
            $listaSede = ArrayHelper::map(Sede::find()->where(['idSede' => Yii::$app->user->identity->datos->sede->idSede])->all(), "idSede", "sede");
        }
        $listaSexo = ArrayHelper::map(Sexo::find()->all(), "idSexo", "sexo");
        $listaEstado = ArrayHelper::map(Estado::find()->all(), "idEstado", "estado");
        $listaNivelEducativo = ArrayHelper::map(NivelEducativo::find()->all(), "idNivelEducativo", "nivelEducativo");
        $listaMotivoNoConcluyo = ArrayHelper::map(MotivoNoConcluyo::find()->all(), "idMotivoNoConcluyo", "motivoNoConcluyo");
        $listaAntecedenteAcademico = ArrayHelper::map(AntecedenteAcademico::find()->all(), "idAntecedenteAcademico", "antecedenteAcademico");

        $ids = null;

        if($idRolUsuarioRecord == 1)
            $ids = [2,5,6];
        elseif($idRolUsuarioRecord == 3)
            $ids = [2,4,6];

        $listaEstatusSustentante = ArrayHelper::map(EstatusSustentante::find()->where(['in', 'idEstatusSustentante', $ids])->all(), "idEstatusSustentante", "estatusSustentante");

        $model->idSede = $model->usuario->idSede;
        $model->nombre = $model->usuario->nombre;
        $model->primerApellido = $model->usuario->primerApellido;
        $model->segundoApellido = $model->usuario->segundoApellido;
        $model->correoElectronico = $model->usuario->usuario;

        $personaAutorizada1Archivo = Yii::getAlias('@web') . '/sustentante/descargar/?id=' . $idSustentante .'&tipoArchivo=' . $model->archivoPersonaAutorizada1 . '&tipo=imagen';
        $personaAutorizada2Archivo = Yii::getAlias('@web') . '/sustentante/descargar/?id=' . $idSustentante .'&tipoArchivo=' . $model->archivoPersonaAutorizada2 . '&tipo=imagen';
        $actaNacimientoArchivo = Yii::getAlias('@web') . '/sustentante/descargar/?id=' . $idSustentante .'&tipoArchivo=' . $model->archivoActaNacimiento . '&tipo=imagen';
        $curpArchivo = Yii::getAlias('@web') . '/sustentante/descargar/?id=' . $idSustentante .'&tipoArchivo=' . $model->archivoCurp . '&tipo=imagen';
        $ineArchivo = Yii::getAlias('@web') . '/sustentante/descargar/?id=' . $idSustentante .'&tipoArchivo=' . $model->archivoIne . '&tipo=imagen';
        $certificadoArchivo = Yii::getAlias('@web') . '/sustentante/descargar/?id=' . $idSustentante .'&tipoArchivo=' . $model->archivoCertificado . '&tipo=imagen';
        $antecedenteSecundariaArchivo = Yii::getAlias('@web') . '/sustentante/descargar/?id=' . $idSustentante .'&tipoArchivo=' . $model->archivoAntecedenteSecundaria . '&tipo=imagen';
        $curriculumArchivo = Yii::getAlias('@web') . '/sustentante/descargar/?id=' . $idSustentante .'&tipoArchivo=' . $model->archivoCurriculum . '&tipo=imagen';
        $doctoUltimoGradoEstudiosArchivo = Yii::getAlias('@web') . '/sustentante/descargar/?id=' . $idSustentante .'&tipoArchivo=' . $model->archivoDoctoUltimoGradoEstudios . '&tipo=imagen';
        $cartaAutorizacionArchivo = Yii::getAlias('@web') . '/sustentante/descargar/?id=' . $idSustentante .'&tipoArchivo=' . $model->archivoCartaAutorizacion . '&tipo=imagen';


        $idEstatusSustentanteAnterior = $model->idEstatusSustentante;

        if($model->load(Yii::$app->request->post()) && Yii::$app->request->isAjax) {
            Yii::$app->response->format = Response::FORMAT_JSON;
            return ActiveForm::validate($model);
        }

        if($model->load(Yii::$app->request->post())) {
            try {
                if(Sustentante::verificarExistenciaCurp($model->curp, $model->idSustentante))
                    throw new \Exception("Ya existe un sustentante registrado con esa CURP");

                if(Sustentante::verificarFechaNacimientoMenorAFechaActual($model->fechaNacimiento))
                    throw new \Exception("Fecha de nacimiento debe ser menor a la fecha actual");

                if(Usuario::verificarExistencia($model->usuario->usuario, $model->idUsuario))
                    throw new \Exception("Ya existe un sustentante registrado con ese correo");

                $transaccion = Yii::$app->db->beginTransaction();

                if($idEstatusSustentanteAnterior == null && $model->idEstatusSustentate == 5) {
                    $model->mesNumeroControl = Sustentante::obtenerLetasDelMes(date("m"));
                    $model->anioNumeroControl = date("y");
                    $model->consecutivoNumeroControl = Sustentante::generarConsecutivo($model->anioNumeroControl);
                    $model->numeroControl = $model->anioNumeroControl . $model->mesNumeroControl . Sustentante::formatearConsecutivo($model->consecutivoNumeroControl);
                }

                $model->actualizar($idUsuarioRecord, $idRolUsuarioRecord);
                $transaccion->commit();

                $mensaje =  "SE ACTUALIZÓ EL SUSTENTANTE CON ID: " . $model->idSustentante;
                $this->logu->registrar($idUsuarioRecord, $idRolUsuarioRecord, $mensaje, 1);
                $this->logs->registrar($idUsuarioRecord, $idRolUsuarioRecord, "DEBUG", $this->controlador, $mensaje);

                Yii::$app->session->setFlash('success', "Se ha actualizado el sustentante");
                return $this->redirect(Yii::getAlias('@web') . '/sustentante/');

            } catch(\Exception $ex) {
                if(isset($transaccion))
                    $transaccion->rollBack();

                $mensaje =  "NO SE ACTUALIZÓ EL SUSTENTANTE CON ID: " . $model->idSustentante;
                $this->logu->registrar($idUsuarioRecord, $idRolUsuarioRecord, $mensaje, 0);
                $this->logs->registrar($idUsuarioRecord, $idRolUsuarioRecord, "ERROR", $this->controlador, $ex->getMessage());

                Yii::$app->session->setFlash('error', "Error al actualizar el sustentante:<br />" . $ex->getMessage());
            }
        }

        return $this->render('validar', [
            'model' => $model,
            'listaAreaConocimiento' => $listaAreaConocimiento,
            'listaPerfilProfesional' => $listaPerfilProfesional,
            'listaSede' => $listaSede,
            'listaSexo' => $listaSexo,
            'listaEstado' => $listaEstado,
            'listaNivelEducativo' => $listaNivelEducativo,
            'listaMotivoNoConcluyo' => $listaMotivoNoConcluyo,
            'listaAntecedenteAcademico' => $listaAntecedenteAcademico,
            'listaEstatusSustentante' => $listaEstatusSustentante,
            'personaAutorizada1Archivo' => $personaAutorizada1Archivo,
            'personaAutorizada2Archivo' => $personaAutorizada2Archivo,
            'actaNacimientoArchivo' => $actaNacimientoArchivo,
            'curpArchivo' => $curpArchivo,
            'ineArchivo' => $ineArchivo,
            'certificadoArchivo' => $certificadoArchivo,
            'antecedenteSecundariaArchivo' => $antecedenteSecundariaArchivo,
            'curriculumArchivo' => $curriculumArchivo,
            'doctoUltimoGradoEstudiosArchivo' => $doctoUltimoGradoEstudiosArchivo,
            'cartaAutorizacionArchivo' => $cartaAutorizacionArchivo
        ]);
    }

    /**
     * Muestra la pantalla de ver de un rol de usuario.
     *
     * @return string|Response
     */
    public function actionVer() {
        $this->layout = "dashboard";
        $idSustentante = trim(Yii::$app->request->get("id"));
        $model = Sustentante::findOne($idSustentante);
        $model->getUsuario();

        $listaAreaConocimiento = ArrayHelper::map(AreaConocimiento::find()->all(), "idAreaConocimiento", "areaConocimiento");
        $modelPerfilProfesional = PerfilProfesional::findOne($model->idPerfilProfesional);
        $listaPerfilProfesional = [];
        if($modelPerfilProfesional != null) {
            $listaPerfilProfesional = ArrayHelper::map(PerfilProfesional::find()->where(['idAreaConocimiento' => $modelPerfilProfesional->idAreaConocimiento])->all(), "idPerfilProfesional", "perfilProfesional");
            $model->idAreaConocimiento = $modelPerfilProfesional->idAreaConocimiento;
        }

        if(Yii::$app->user->identity->rolUsuario->idRolUsuario != 3) {
            $listaSede = ArrayHelper::map(Sede::find()->all(), "idSede", "sede");
        }else {
            $listaSede = ArrayHelper::map(Sede::find()->where(['idSede' => Yii::$app->user->identity->datos->sede->idSede])->all(), "idSede", "sede");
        }
        $listaSexo = ArrayHelper::map(Sexo::find()->all(), "idSexo", "sexo");
        $listaEstado = ArrayHelper::map(Estado::find()->all(), "idEstado", "estado");
        $listaNivelEducativo = ArrayHelper::map(NivelEducativo::find()->all(), "idNivelEducativo", "nivelEducativo");
        $listaMotivoNoConcluyo = ArrayHelper::map(MotivoNoConcluyo::find()->all(), "idMotivoNoConcluyo", "motivoNoConcluyo");
        $listaAntecedenteAcademico = ArrayHelper::map(AntecedenteAcademico::find()->all(), "idAntecedenteAcademico", "antecedenteAcademico");

        $model->nombre = $model->usuario->nombre;
        $model->primerApellido = $model->usuario->primerApellido;
        $model->segundoApellido = $model->usuario->segundoApellido;
        $model->correoElectronico = $model->usuario->usuario;

        $personaAutorizada1Archivo = Yii::getAlias('@web') . '/sustentante/descargar/?id=' . $idSustentante .'&tipoArchivo=' . $model->archivoPersonaAutorizada1 . '&tipo=imagen';
        $personaAutorizada2Archivo = Yii::getAlias('@web') . '/sustentante/descargar/?id=' . $idSustentante .'&tipoArchivo=' . $model->archivoPersonaAutorizada2 . '&tipo=imagen';
        $actaNacimientoArchivo = Yii::getAlias('@web') . '/sustentante/descargar/?id=' . $idSustentante .'&tipoArchivo=' . $model->archivoActaNacimiento . '&tipo=imagen';
        $curpArchivo = Yii::getAlias('@web') . '/sustentante/descargar/?id=' . $idSustentante .'&tipoArchivo=' . $model->archivoCurp . '&tipo=imagen';
        $ineArchivo = Yii::getAlias('@web') . '/sustentante/descargar/?id=' . $idSustentante .'&tipoArchivo=' . $model->archivoIne . '&tipo=imagen';
        $certificadoArchivo = Yii::getAlias('@web') . '/sustentante/descargar/?id=' . $idSustentante .'&tipoArchivo=' . $model->archivoCertificado . '&tipo=imagen';
        $antecedenteSecundariaArchivo = Yii::getAlias('@web') . '/sustentante/descargar/?id=' . $idSustentante .'&tipoArchivo=' . $model->archivoAntecedenteSecundaria . '&tipo=imagen';
        $curriculumArchivo = Yii::getAlias('@web') . '/sustentante/descargar/?id=' . $idSustentante .'&tipoArchivo=' . $model->archivoCurriculum . '&tipo=imagen';
        $doctoUltimoGradoEstudiosArchivo = Yii::getAlias('@web') . '/sustentante/descargar/?id=' . $idSustentante .'&tipoArchivo=' . $model->archivoDoctoUltimoGradoEstudios . '&tipo=imagen';
        $cartaAutorizacionArchivo = Yii::getAlias('@web') . '/sustentante/descargar/?id=' . $idSustentante .'&tipoArchivo=' . $model->archivoCartaAutorizacion . '&tipo=imagen';

        if($model == null)
            return $this->redirect(Yii::getAlias('@web') . '/sustentante/');
    
        return $this->render('ver', [
            'model' => $model,
            'listaAreaConocimiento' => $listaAreaConocimiento,
            'listaPerfilProfesional' => $listaPerfilProfesional,
            'listaSede' => $listaSede,
            'listaSexo' => $listaSexo,
            'listaEstado' => $listaEstado,
            'listaNivelEducativo' => $listaNivelEducativo,
            'listaMotivoNoConcluyo' => $listaMotivoNoConcluyo,
            'listaAntecedenteAcademico' => $listaAntecedenteAcademico,
            'personaAutorizada1Archivo' => $personaAutorizada1Archivo,
            'personaAutorizada2Archivo' => $personaAutorizada2Archivo,
            'actaNacimientoArchivo' => $actaNacimientoArchivo,
            'curpArchivo' => $curpArchivo,
            'ineArchivo' => $ineArchivo,
            'certificadoArchivo' => $certificadoArchivo,
            'antecedenteSecundariaArchivo' => $antecedenteSecundariaArchivo,
            'curriculumArchivo' => $curriculumArchivo,
            'doctoUltimoGradoEstudiosArchivo' => $doctoUltimoGradoEstudiosArchivo,
            'cartaAutorizacionArchivo' => $cartaAutorizacionArchivo
        ]);
    }

    /**
     * Método que elimina un rol de usuario.
     *
     * @return Response
     */
    public function actionEliminar() {
        $idSustentante = trim(Yii::$app->request->get("id"));
        $model = Sustentante::findOne($idSustentante);

        if($model == null)
            return $this->redirect(Yii::getAlias('@web') . '/sustentante/');

        try {
            $idUsuarioDelete = Yii::$app->user->identity->idUsuario;
            $idRolUsuarioDelete = Yii::$app->user->identity->rolUsuario->idRolUsuario;

            $transaccion = Yii::$app->db->beginTransaction();
            $modelUsuario = $model->usuario;
            $model->eliminar($idUsuarioDelete, $idRolUsuarioDelete);
            $modelUsuario->eliminar($idUsuarioDelete, $idRolUsuarioDelete);
            $transaccion->commit();

            $mensaje =  "SE ELIMINÓ EL SUSTENTANTE CON ID: " . $idSustentante;
            $this->logu->registrar($idUsuarioDelete, $idRolUsuarioDelete, $mensaje, 1);
            $this->logs->registrar($idUsuarioDelete, $idRolUsuarioDelete, "DEBUG", $this->controlador, $mensaje);

            Yii::$app->session->setFlash('success', "Se ha eliminado el sustentante");
        } catch(IntegrityException $ex) {
            if(isset($transaccion))
                $transaccion->rollBack();

            $mensaje =  "NO SE ELIMINÓ EL SUSTENTANTE CON ID: " . $idSustentante;
            $this->logu->registrar($idUsuarioDelete, $idRolUsuarioDelete, $mensaje, 0);
            $this->logs->registrar($idUsuarioDelete, $idRolUsuarioDelete, "ERROR", $this->controlador, $ex->getMessage());

            Yii::$app->session->setFlash('error', "Error al eliminar el sustentante:<br />" .
                "Existen datos relacionados al sustentante: " . $model->rolUsuario);
        } catch(\Exception $ex) {
            if(isset($transaccion))
                $transaccion->rollBack();

            $mensaje =  "NO SE ELIMINÓ EL SUSTENTANTE CON ID: " . $idSustentante;
            $this->logu->registrar($idUsuarioDelete, $idRolUsuarioDelete, $mensaje, 0);
            $this->logs->registrar($idUsuarioDelete, $idRolUsuarioDelete, "ERROR", $this->controlador, $ex->getMessage());

            Yii::$app->session->setFlash('error', "Error al eliminar el sustentante:<br />" . $ex->getMessage());
        }

        return $this->redirect(Yii::getAlias('@web') . '/sustentante/');
    }

    /**
     * Método que descarga el archivos de aviso.
     *
     * @return Response
     */
    public function actionDescargar() {
        
        $idSustentante = trim(Yii::$app->request->get("id"));
        $tipoArchivo = trim(Yii::$app->request->get("tipoArchivo"));
        $model = Sustentante::findOne($idSustentante);
        
        if($model == null)
            return $this->redirect(Yii::getAlias('@web') . '/sustentante/');
            
            $rutaImagenWeb = null;
            
            if(empty($tipoArchivo)){
                $rutaImagenWeb = null;
            } else {
                $rutaImagenWeb = Yii::$app->basePath . "/" . Yii::$app->params['ruta-archivos-sustentante'] . $tipoArchivo;
            }
            if(file_exists($rutaImagenWeb)) {
                Utils::descargarArchivo($rutaImagenWeb);
            }
    }

    /**
     * Método que devuelve por petición de Ajax los perfiles profesionales por un área de conocimiento.
     */
    public function actionJsonPerfilesProfesionalesXAreaConocimiento()
    {
        Yii::$app->response->format = Response::FORMAT_JSON;
        $estatus = "error";
        $mensaje = "Ocurrió un error inesperado, por favor intentalo nuevamente";
        $datos = null;
        $response = null;

        if(Yii::$app->request->isAjax)
        {
            if($request = Yii::$app->request->post())
            {
                try
                {
                    $perfilProfesional = PerfilProfesional::find()->where(['idAreaConocimiento' => $request['idAreaConocimiento']])->all();
                    foreach($perfilProfesional as $item)
                    {
                        $datos[] =
                            [
                                'idPerfilProfesional' => $item->idPerfilProfesional,
                                'perfilProfesional' => $item->perfilProfesional
                            ];
                    }

                    $estatus = "ok";
                    $mensaje = "";
                }
                catch(\Exception $ex)
                {
                    $estatus = "error";
                    $mensaje = $ex->getMessage();
                }
            }
        }

        echo Json::encode($response = [
            'estatus' => $estatus,
            'mensaje' => $mensaje,
            'datos' => $datos,
        ]);
        exit(0);
    }

    /**
     * Método que exporta un archivo de los sustentantes.
     *
     * @return Response
     */
    public function actionExportar() {
        $registros = null;
        $sustentantes = Sustentante::find()->orderBy('createDate')->all();

        header('Content-Description: File Transfer');
        header('Content-Type: application/octet-stream');
        header('Content-Disposition: attachment; filename="sustentantes.csv');
        header('Cache-Control: no-cache');
        header("Content-Transfer-Encoding: binary");

        if($sustentantes != null) {
            $campos = Sustentante::instance()->attributeLabels();
            $areaConocimientoCampo = AreaConocimiento::instance()->getAttributeLabel('areaConocimiento');
            $sedeCampo = Sede::instance()->getAttributeLabel('sede');
            $usuarioCampo = Usuario::instance()->getAttributeLabel('usuario');
            $nombreCampo = Usuario::instance()->getAttributeLabel('nombre');
            $primerApellidoCampo = Usuario::instance()->getAttributeLabel('primerApellido');
            $segundoApellidoCampo = Usuario::instance()->getAttributeLabel('segundoApellido');

            $registros =
                '"' . $campos['idSustentante'] . '",' .
                '"' . $campos['idSexo'] . '",' .
                '"' . $campos['idEstado'] . '",' .
                '"' . $campos['idPerfilProfesional'] . '",' .
                '"' . $areaConocimientoCampo . '",' .
                '"' . $campos['idMotivoNoConcluyo'] . '",' .
                '"' . $campos['otroMotivo'] . '",' .
                '"' . $campos['idAntecedenteAcademico'] . '",' .
                '"' . $campos['idEstadoRadica'] . '",' .
                '"' . $campos['idNivelEducativo'] . '",' .
                '"' . $campos['idEstatusSustentante'] . '",' .
                '"' . $sedeCampo . '",' .
                '"' . $usuarioCampo . '",' .
                '"' . $nombreCampo . '",' .
                '"' . $primerApellidoCampo . '",' .
                '"' . $segundoApellidoCampo . '",' .
                '"' . $campos['numeroControl'] . '",' .
                '"' . $campos['curp'] . '",' .
                '"' . $campos['fechaNacimiento'] . '",' .
                '"' . $campos['nacionalidad'] . '",' .
                '"' . $campos['lugarNacimiento'] . '",' .
                '"' . $campos['calleNumero'] . '",' .
                '"' . $campos['colonia'] . '",' .
                '"' . $campos['codigoPostal'] . '",' .
                '"' . $campos['municipio'] . '",' .
                '"' . $campos['telefonoCasa'] . '",' .
                '"' . $campos['telefonoOficina'] . '",' .
                '"' . $campos['celular'] . '",' .
                '"' . $campos['institucion'] . '",' .
                '"' . $campos['porcentajeCreditosCursados'] . '",' .
                '"' . $campos['licenciatura'] . '",' .
                '"' . $campos['porcentajeCreditosObtenidos'] . '",' .
                '"' . $campos['personaAutorizada1'] . '",' .
                '"' . $campos['archivoPersonaAutorizada1'] . '",' .
                '"' . $campos['personaAutorizada2'] . '",' .
                '"' . $campos['archivoPersonaAutorizada2'] . '",' .
                '"' . $campos['archivoActaNacimiento'] . '",' .
                '"' . $campos['archivoCurp'] . '",' .
                '"' . $campos['archivoIne'] . '",' .
                '"' . $campos['archivoCertificado'] . '",' .
                '"' . $campos['archivoAntecedenteSecundaria'] . '",' .
                '"' . $campos['archivoCurriculum'] . '",' .
                '"' . $campos['archivoDoctoUltimoGradoEstudios'] . '",' .
                '"' . $campos['archivoCartaAutorizacion'] . '",' .
                '"Fecha de registro"'.
                "\n";

            foreach($sustentantes as $item) {
                $registros .= '"' . $item->idSustentante . '",';
                $registros .= '"' . $item->sexo->sexo . '",';
                $registros .= '"' . $item->estado->estado . '",';

                if(isset($item->perfilProfesional)) {
                    $registros .= '"' . $item->perfilProfesional->perfilProfesional . '",';
                    $registros .= '"' . $item->perfilProfesional->areaConocimiento->areaConocimiento . '",';
                } else {
                    $registros .= '"",';
                    $registros .= '"",';
                }

                if(isset($item->motivoNoConcluyo))
                    $registros .= '"' . $item->motivoNoConcluyo->motivoNoConcluyo . '",';
                else
                    $registros .= '"",';

                $registros .= '"' . $item->otroMotivo . '",';

                if(isset($item->antecedenteAcademico))
                    $registros .= '"' . $item->antecedenteAcademico->antecedenteAcademico . '",';
                else
                    $registros .= '"",';

                if(isset($item->estadoRadica))
                    $registros .= '"' . $item->estadoRadica->estado . '",';
                else
                    $registros .= '"",';

                if(isset($item->nivelEducativo))
                    $registros .= '"' . $item->nivelEducativo->nivelEducativo . '",';
                else
                    $registros .= '"",';

                if(isset($item->estatusSustentante))
                    $registros .= '"' . $item->estatusSustentante->estatusSustentante . '",';
                else
                    $registros .= '"",';

                if(isset($item->usuario)) {
                    if(isset($item->usuario->sede))
                        $registros .= '"' . $item->usuario->sede->sede . '",';
                    else
                        $registros .= '"",';

                    $registros .= '"' . $item->usuario->usuario . '",';
                    $registros .= '"' . $item->usuario->nombre . '",';
                    $registros .= '"' . $item->usuario->primerApellido . '",';
                    $registros .= '"' . $item->usuario->segundoApellido . '",';
                } else {
                    $registros .= '"",';
                    $registros .= '"",';
                    $registros .= '"",';
                    $registros .= '"",';
                }

                $registros .= '"' . $item->numeroControl . '",';
                $registros .= '"' . $item->curp . '",';
                $registros .= '"' . $item->fechaNacimiento . '",';
                $registros .= '"' . $item->nacionalidad . '",';
                $registros .= '"' . $item->lugarNacimiento . '",';
                $registros .= '"' . $item->calleNumero . '",';
                $registros .= '"' . $item->colonia . '",';
                $registros .= '"' . $item->codigoPostal . '",';
                $registros .= '"' . $item->municipio . '",';
                $registros .= '"' . $item->telefonoCasa . '",';
                $registros .= '"' . $item->telefonoOficina . '",';
                $registros .= '"' . $item->celular . '",';
                $registros .= '"' . $item->institucion . '",';
                $registros .= '"' . $item->porcentajeCreditosCursados . '",';
                $registros .= '"' . $item->licenciatura . '",';
                $registros .= '"' . $item->porcentajeCreditosObtenidos . '",';
                $registros .= '"' . $item->personaAutorizada1 . '",';
                $registros .= '"' . $item->archivoPersonaAutorizada1 . '",';
                $registros .= '"' . $item->personaAutorizada2 . '",';
                $registros .= '"' . $item->archivoPersonaAutorizada2 . '",';
                $registros .= '"' . $item->archivoActaNacimiento . '",';
                $registros .= '"' . $item->archivoCurp . '",';
                $registros .= '"' . $item->archivoIne . '",';
                $registros .= '"' . $item->archivoCertificado . '",';
                $registros .= '"' . $item->archivoAntecedenteSecundaria . '",';
                $registros .= '"' . $item->archivoCurriculum . '",';
                $registros .= '"' . $item->archivoDoctoUltimoGradoEstudios . '",';
                $registros .= '"' . $item->archivoCartaAutorizacion . '",';
                $registros .= '"' . $item->createDate . '"';
                $registros .= "\n";

            }

            //echo $registros;
        }

        return $registros;
    }
}


?>