<?php
/*
 * UsuarioController.php
 * Springfield
 * Copyright ©: 2022 Integrasoft. Todos los derechos reservados.
 * Desarrollador(es):
 *      L.I. Humberto Javier Flores Tirado.
 * Fecha de creación: 25/03/2022.
 * Fecha de modificación: 17/05/2022.
 * Descripción:
 */

namespace app\controllers;

use Yii;
use yii\filters\AccessControl;
use yii\web\Controller;
use yii\web\Response;
use yii\filters\VerbFilter;
use yii\widgets\ActiveForm;
use yii\helpers\ArrayHelper;
use yii\db\IntegrityException;
use yii\helpers\Json;

use app\models\estado\Estado;
use app\models\log\BitacoraUsuario;
use app\models\log\BitacoraSistema;

use app\models\usuario\Usuario;
use app\models\usuario\UsuarioSearch;
use app\models\sesion\UsuarioSesion;
use app\models\rolusuario\RolUsuario;
use app\models\sesion\EstatusUsuario;
use app\models\sede\Sede;

use yii\db\Expression;

/**
 * Clase controlador de usuarios.
 *
 * Class UsuarioController
 * @package app\controllers
 */
class UsuarioController extends Controller {
    private $controlador;
    private $logu;
    private $logs;

    /**
     * Método que restringue el acceso de las acciones de acuerdo a los roles.
     *
     * @return array
     */
    public function behaviors() {
        return [
            'access' => [
                'class' => AccessControl::className(),
                'only' => ['index', 'registrar', 'actualizar', 'ver', 'eliminar', 'actualizar-datos'],
                'rules' => [
                    [
                        'actions' => ['index', 'registrar', 'actualizar', 'ver', 'eliminar', 'actualizar-datos'],
                        'allow' => true,
                        'roles' => ['@'],
                        'matchCallback' => function($rule, $action) {
                            $permiso = null;

                            switch($action->id) {
                                case 'index':
                                    $permiso = 'Buscar usuario';
                                    break;

                                case 'registrar':
                                    $permiso = 'Registrar usuario';
                                    break;

                                case 'actualizar':
                                    $permiso = 'Actualizar usuario';
                                    break;

                                case 'ver':
                                    $permiso = 'Ver usuario';
                                    break;

                                case 'eliminar':
                                    $permiso = 'Eliminar usuario';
                                    break;

                                case 'actualizar-datos':
                                    $permiso = 'Actualizar datos';
                                    break;

                                default:
                                    return false;
                            }

                            if(isset(Yii::$app->user->identity))
                                return Yii::$app->user->identity->datos->verificarPermisos($permiso);
                            else
                                return false;
                        }
                    ],
                ],
            ],
            'verbs' => [
                'class' => VerbFilter::className(),
                'actions' => [
                    'logout' => ['post'],
                ],
            ],
        ];
    }

    /**
     * @inheritdoc
     */
    public function actions() {
        $this->layout = "dashboard";
        $this->controlador = "UsuarioController";
        $this->logu = new BitacoraUsuario();
        $this->logs = new BitacoraSistema();

        return [
            'error' => [
                'class' => 'yii\web\ErrorAction',
            ],
            'captcha' => [
                'class' => 'yii\captcha\CaptchaAction',
                'fixedVerifyCode' => YII_ENV_TEST ? 'testme' : null,
            ],

        ];
    }

    public function actionStatus() {}

    /**
     * Muestra la pantalla de gestión de usuarios.
     */
    public function actionIndex() {
        $searchModel = new UsuarioSearch();
        $searchModel->idUsuario = Yii::$app->user->identity->idUsuario;

        $listaRolesUsuario = ArrayHelper::map(RolUsuario::find()->where(['!=', 'idRolUsuario', [2,3]])->orderBy('rolUsuario')->all(), "idRolUsuario", "rolUsuario");
        $listaEstatusUsuario = ArrayHelper::map(EstatusUsuario::find()->where(['<', 'idEstatusUsuario', 3])->orderBy('estatusUsuario')->all(), "idEstatusUsuario", "estatusUsuario");
        $listaEstados = ArrayHelper::map(Estado::find()->orderBy('estado')->all(), "idEstado", "estado");
        $listaSedes = ArrayHelper::map(Sede::find()->orderBy('sede')->all(), "idSede", "sede");

        $dataProvider = $searchModel->search(Yii::$app->request->queryParams);

        return $this->render('index', [
            'searchModel' => $searchModel,
            'dataProvider' => $dataProvider,
            'listaEstados' => $listaEstados,
            'listaRolesUsuario' => $listaRolesUsuario,
            'listaEstatusUsuario' => $listaEstatusUsuario,
            'listaSedes' => $listaSedes,
        ]);
    }

    /**
     * Muestra la pantalla de registro de un usuario, además realiza el registro.
     *
     * @return array|string
     */
    public function actionRegistrar() {
        $model = new Usuario();
        $model->scenario = "registrar";

        $listaRolesUsuario = ArrayHelper::map(RolUsuario::find()->where(['IN', 'idRolUsuario', [1, 3]])->orderBy('rolUsuario')->all(), "idRolUsuario", "rolUsuario");
        $listaEstatusUsuario = ArrayHelper::map(EstatusUsuario::find()->where(['<', 'idEstatusUsuario', 3])->orderBy('estatusUsuario')->all(), "idEstatusUsuario", "estatusUsuario");
        $listaEstados = ArrayHelper::map(Estado::find()->orderBy('estado')->all(), "idEstado", "estado");
        $listaSedes = ArrayHelper::map(Sede::find()->orderBy('sede')->all(), "idSede", "sede");

        if($model->load(Yii::$app->request->post()) && Yii::$app->request->isAjax) {
            Yii::$app->response->format = Response::FORMAT_JSON;
            return ActiveForm::validate($model);
        }

        if($model->load(Yii::$app->request->post())) {
            $idUsuarioCreate = Yii::$app->user->identity->idUsuario;
            $idRolUsuarioCreate = Yii::$app->user->identity->rolUsuario->idRolUsuario;

            try {
                if(UsuarioSesion::verificarExistencia($model->usuario))
                    throw new \Exception("Ya existe un usuario registrado con ese correo electrónico");

                $model->registrar($idUsuarioCreate, $idRolUsuarioCreate);

                $mensaje =  "SE REGISTRÓ EL USUARIO CON ID: " . $model->idUsuario;
                $this->logu->registrar($idUsuarioCreate, $idRolUsuarioCreate, $mensaje, 1);
                $this->logs->registrar($idUsuarioCreate, $idRolUsuarioCreate, "DEBUG", $this->controlador, $mensaje);

                Yii::$app->session->setFlash('success', "Se ha registrado el usuario");

                $model->idRolUsuario = null;
                $model->idEstatusUsuario = null;
                $model->usuario = null;
                $model->contrasenia = null;
                $model->confirmaContrasenia = null;
                $model->nombre = null;
                $model->primerApellido = null;
                $model->segundoApellido = null;
                $model->idEstado = null;
            } catch(\Exception $ex) {
                $mensaje =  "NO SE REGISTRÓ EL USUARIO CON USUARIO: " . $model->usuario;
                $this->logu->registrar($idUsuarioCreate, $idRolUsuarioCreate, $mensaje, 0);
                $this->logs->registrar($idUsuarioCreate, $idRolUsuarioCreate, "ERROR", $this->controlador, $ex->getMessage());

                Yii::$app->session->setFlash('error', "Error al registrar el usuario:<br />" . $ex->getMessage());
            }
        }

        return $this->render('registrar', [
            'model' => $model,
            'listaRolesUsuario' => $listaRolesUsuario,
            'listaEstatusUsuario' => $listaEstatusUsuario,
            'listaEstados' => $listaEstados,
            'listaSedes' => $listaSedes,
        ]);
    }

    /**
     * Muestra la pantalla de actualización de un usuario, además realiza el registro.
     *
     * @return array|string
     */
    public function actionActualizar() {
        $idUsuario = trim(Yii::$app->request->get("id"));
        $model = Usuario::findOne($idUsuario);

        if($model == null)
            return $this->redirect(Yii::getAlias('@web') . '/usuario/');

        $listaRolesUsuario = ArrayHelper::map(RolUsuario::find()->where(['IN', 'idRolUsuario', [1 ,3]])->orderBy('rolUsuario')->all(), "idRolUsuario", "rolUsuario");
        $listaEstatusUsuario = ArrayHelper::map(EstatusUsuario::find()->where(['<', 'idEstatusUsuario', 3])->orderBy('estatusUsuario')->all(), "idEstatusUsuario", "estatusUsuario");
        $listaEstados = ArrayHelper::map(Estado::find()->orderBy('estado')->all(), "idEstado", "estado");
        $listaSedes = ArrayHelper::map(Sede::find()->orderBy('sede')->all(), "idSede", "sede");

        $contraseniaActual = $model->contrasenia;
        $model->contrasenia = "";
        $model->scenario = "actualizar";

        if($model->load(Yii::$app->request->post()) && Yii::$app->request->isAjax) {
            Yii::$app->response->format = Response::FORMAT_JSON;
            return ActiveForm::validate($model);
        }

        if($model->load(Yii::$app->request->post())) {
            $idUsuarioRecord = Yii::$app->user->identity->idUsuario;
            $idRolUsuarioRecord = Yii::$app->user->identity->rolUsuario->idRolUsuario;

            try {
                if(UsuarioSesion::verificarExistencia($model->usuario, $model->idUsuario))
                    throw new \Exception("Ya existe un usuario registrado con ese correo electrónico");

                if(empty($model->contrasenia))
                    $model->contrasenia = $contraseniaActual;
                else
                    $model->contrasenia = md5($model->contrasenia);

                $model->actualizar($idUsuarioRecord, $idRolUsuarioRecord);

                $mensaje =  "SE ACTUALIZÓ EL USUARIO CON ID: " . $model->idUsuario;
                $this->logu->registrar($idUsuarioRecord, $idRolUsuarioRecord, $mensaje, 1);
                $this->logs->registrar($idUsuarioRecord, $idRolUsuarioRecord, "DEBUG", $this->controlador, $mensaje);

                Yii::$app->session->setFlash('success', "Se ha actualizado el usuario");

                $model->contrasenia = null;
                $model->confirmaContrasenia = null;
            } catch(\Exception $ex) {
                $mensaje =  "NO SE ACTUALIZÓ EL USUARIO CON ID: " . $model->idUsuario;
                $this->logu->registrar($idUsuarioRecord, $idRolUsuarioRecord, $mensaje, 0);
                $this->logs->registrar($idUsuarioRecord, $idRolUsuarioRecord, "ERROR", $this->controlador, $ex->getMessage());

                Yii::$app->session->setFlash('error', "Error al actualizar el usuario:<br />" . $ex->getMessage());
            }
        }

        return $this->render('actualizar', [
            'model' => $model,
            'listaRolesUsuario' => $listaRolesUsuario,
            'listaEstatusUsuario' => $listaEstatusUsuario,
            'listaEstados' => $listaEstados,
            'listaSedes' => $listaSedes,
        ]);
    }

    /**
     * Muestra la pantalla de actualización de los datos del usuario, además realiza la actualización.
     *
     * @return array|string|Response
     */
    public function actionActualizarDatos() {
        $idUsuario = Yii::$app->user->identity->idUsuario;
        $model = Usuario::findOne($idUsuario);
        //$model->scenario = "actualizar"; //SI SE DESCOMENTA EL ESCENARIO, NO ACTUALIZA LA INFORMACIÓN

        if($model == null)
            return $this->redirect('/sesion/');

        $listaEstados = ArrayHelper::map(Estado::find()->orderBy('estado')->all(), "idEstado", "estado");

        if($model->load(Yii::$app->request->post()) && Yii::$app->request->isAjax) {
            Yii::$app->response->format = Response::FORMAT_JSON;
            return ActiveForm::validate($model);
        }

        if($model->load(Yii::$app->request->post())) {
            $idUsuarioRecord = Yii::$app->user->identity->idUsuario;
            $idRolUsuarioRecord = Yii::$app->user->identity->rolUsuario->idRolUsuario;

            try {
                if(UsuarioSesion::verificarExistencia($model->usuario, $model->idUsuario))
                    throw new \Exception("Ya existe un usuario registrado con ese nombre");

                //echo $model->segundoApellido;
                //exit;

                $model->actualizar($idUsuarioRecord, $idRolUsuarioRecord);

                $mensaje =  "SE ACTUALIZÓ EL USUARIO CON ID: " . $model->idUsuario;
                $this->logu->registrar($idUsuarioRecord, $idRolUsuarioRecord, $mensaje, 1);
                $this->logs->registrar($idUsuarioRecord, $idRolUsuarioRecord, "DEBUG", $this->controlador, $mensaje);

                Yii::$app->session->setFlash('success', "Se han actualizado los datos");
            } catch(\Exception $ex) {
                $mensaje =  "NO SE ACTUALIZÓ EL USUARIO CON ID: " . $model->idUsuario;
                $this->logu->registrar($idUsuarioRecord, $idRolUsuarioRecord, $mensaje, 0);
                $this->logs->registrar($idUsuarioRecord, $idRolUsuarioRecord, "ERROR", $this->controlador, $ex->getMessage());

                Yii::$app->session->setFlash('error', "Error al actualizar los datos:<br />" . $ex->getMessage());
            }
        }

        return $this->render('actualizar-datos', [
            'model' => $model,
            'listaEstados' => $listaEstados,
        ]);
    }



    /**
     * Muestra la pantalla de ver de un usuario.
     *
     * @return string|Response
     */
    public function actionVer() {
        $idUsuario = trim(Yii::$app->request->get("id"));
        $model = Usuario::findOne($idUsuario);

        if($model == null)
            return $this->redirect(Yii::getAlias('@web') . '/usuario/');

        $listaRolesUsuario = ArrayHelper::map(RolUsuario::find()->where(['IN', 'idRolUsuario', [1, 3]])->orderBy('rolUsuario')->all(), "idRolUsuario", "rolUsuario");
        $listaEstatusUsuario = ArrayHelper::map(EstatusUsuario::find()->where(['<', 'idEstatusUsuario', 3])->orderBy('estatusUsuario')->all(), "idEstatusUsuario", "estatusUsuario");
        $listaEstados = ArrayHelper::map(Estado::find()->orderBy('estado')->all(), "idEstado", "estado");
        $listaSedes = ArrayHelper::map(Sede::find()->orderBy('sede')->all(), "idSede", "sede");

        $model->contrasenia = "";
        $model->confirmaContrasenia = "";

        return $this->render('ver', [
            'model' => $model,
            'listaRolesUsuario' => $listaRolesUsuario,
            'listaEstatusUsuario' => $listaEstatusUsuario,
            'listaEstados' => $listaEstados,
            'listaSedes' => $listaSedes,
        ]);
    }

    /**
     * Método que elimina un usuario.
     *
     * @return Response
     */
    public function actionEliminar() {
        $idUsuario = trim(Yii::$app->request->get("id"));
        $model = Usuario::findOne($idUsuario);

        if($model == null)
            return $this->redirect(Yii::getAlias('@web') . '/usuario/');

        try {
            $idUsuarioDelete = Yii::$app->user->identity->idUsuario;
            $idRolUsuarioDelete = Yii::$app->user->identity->rolUsuario->idRolUsuario;

            $model->eliminar($idUsuarioDelete, $idRolUsuarioDelete);

            $mensaje =  "SE ELIMINÓ EL USUARIO CON ID: " . $idUsuario;
            $this->logu->registrar($idUsuarioDelete, $idRolUsuarioDelete, $mensaje, 1);
            $this->logs->registrar($idUsuarioDelete, $idRolUsuarioDelete, "DEBUG", $this->controlador, $mensaje);

            Yii::$app->session->setFlash('success', "Se ha eliminado el usuario");
        } catch(IntegrityException $ex) {
            $mensaje =  "NO SE ELIMINÓ EL USUARIO CON ID: " . $idUsuario;
            $this->logu->registrar($idUsuarioDelete, $idRolUsuarioDelete, $mensaje, 0);
            $this->logs->registrar($idUsuarioDelete, $idRolUsuarioDelete, "ERROR", $this->controlador, $ex->getMessage());

            Yii::$app->session->setFlash('error', "Error al eliminar el usuario:<br />" .
                "Existen datos relacionados al usuario: " . $model->usuario);
        } catch(\Exception $ex) {
            $mensaje =  "NO SE ELIMINÓ EL USUARIO CON ID: " . $idUsuario;
            $this->logu->registrar($idUsuarioDelete, $idRolUsuarioDelete, $mensaje, 0);
            $this->logs->registrar($idUsuarioDelete, $idRolUsuarioDelete, "ERROR", $this->controlador, $ex->getMessage());

            Yii::$app->session->setFlash('error', "Error al eliminar el usuario:<br />" . $ex->getMessage());
        }

        return $this->redirect(Yii::getAlias('@web') . '/usuario/');
    }

    /**
     * Método que devuelve por petición de Ajax las sedes por un estado.
     */
    public function actionJsonSedesXEstado() {
        Yii::$app->response->format = Response::FORMAT_JSON;
        $estatus = "error";
        $mensaje = "Ocurrió un error inesperado, por favor intentalo nuevamente";
        $datos = null;
        $response = null;

        if(Yii::$app->request->isAjax) {
            if($request = Yii::$app->request->post()) {
                try {
                    $sedes = Sede::find()->where(['idEstado' => $request['idEstado']])->orderBy('sede')->all();

                    if($sedes != null) {
                        foreach($sedes as $item) {
                            $datos[] = [
                                'idSede' => $item->idSede,
                                'sede' => $item->sede,
                            ];
                        }
                    }

                    $estatus = "ok";
                    $mensaje = "";
                }
                catch(\Exception $ex) {
                    $estatus = "error";
                    $mensaje = $ex->getMessage();
                }
            }
        }

        echo Json::encode($response = [
            'estatus' => $estatus,
            'mensaje' => $mensaje,
            'datos' => $datos,
        ]);
        exit(0);
    }
}

?>