<?php
/*
 * RolUsuario.php
 * Springfield
 * Copyright ©: 2022 Integrasoft. Todos los derechos reservados.
 * Desarrollador(es):
 *      L.I. Humberto Javier Flores Tirado.
 * Fecha de creación: 25/03/2022.
 * Fecha de modificación: 25/03/2022.
 * Descripción:
 */

namespace app\models\rolusuario;

use Yii;
use yii\db\ActiveRecord;
use yii\db\Expression;
use app\models\usuario\Usuario;

/**
 * Clase modelo de la tabla "RolUsuario".
 *
 * @property int $idRolUsuario
 * @property string $rolUsuario
 * @property string $createDate
 * @property string $recordDate
 * @property string $deleteDate
 * @property int $idUsuarioCreate
 * @property int $idUsuarioRecord
 * @property int $idUsuarioDelete
 * @property int $idRolUsuarioCreate
 * @property int $idRolUsuarioRecord
 * @property int $idRolUsuarioDelete
 *
 * @property RolUsuarioPermiso[] $rolUsuarioPermisos
 * @property Usuario[] $usuarios
 *
 * @property int $idPermiso
 * @property Int[] $permisos
 */
class RolUsuario extends ActiveRecord {
    public $permisos;

    /**
     * Método que devuelve la tabla de la que es modelo la clase.
     */
    public static function tableName() {
        return 'RolUsuario';
    }

    /**
     * Método que define las validaciones de los atributos.
     */
    public function rules() {
        return [
            ['permisos', 'safe'],
            ['rolUsuario','required','message' => 'Es obligatorio'],
            ['rolUsuario', 'match', 'pattern' => "/^.{3,100}$/", 'message' => 'Debe tener entre 3 y 100 caracteres'],

            [['rolUsuario', 'createDate', 'recordDate', 'idUsuarioCreate', 'idUsuarioRecord', 'idRolUsuarioCreate', 'idRolUsuarioRecord'], 'required'],
            [['createDate', 'recordDate', 'deleteDate'], 'safe'],
            [['idUsuarioCreate', 'idUsuarioRecord', 'idUsuarioDelete', 'idRolUsuarioCreate', 'idRolUsuarioRecord', 'idRolUsuarioDelete'], 'integer'],
            [['rolUsuario'], 'string', 'max' => 100],
        ];
    }

    /**
     * Método que define las etiquetas de los atributos.
     */
    public function attributeLabels() {
        return [
            'idRolUsuario' => 'ID',
            'rolUsuario' => 'Rol de usuario',
            'createDate' => 'Fecha de creación',
            'recordDate' => 'Fecha de modificación',
            'deleteDate' => 'Fecha de eliminación',

            'permisos' => 'Permisos',
        ];
    }

    /**
     * Método que registrar un rol de usuario.
     *
     * @param $idUsuarioCreate
     * @param $idRolUsuarioCreate
     * @throws \Exception
     */
    public function registrar($idUsuarioCreate, $idRolUsuarioCreate) {
        $this->createDate = new Expression('NOW()');
        $this->recordDate = new Expression('NOW()');
        $this->idUsuarioCreate = $idUsuarioCreate;
        $this->idUsuarioRecord = $idUsuarioCreate;
        $this->idRolUsuarioCreate = $idRolUsuarioCreate;
        $this->idRolUsuarioRecord = $idRolUsuarioCreate;

        if(!$this->save(false)) {
            $error = print_r($this->getErrors(), true);
            throw new \Exception($error);
        }
    }

    /**
     * Método que registra la lista de permisos selecciados.
     *
     * @param $permisos
     * @param $idUsuarioCreate
     * @param $idRolUsuarioCreate
     * @param $controlador
     * @param $logu
     * @param $logs
     * @throws \Exception
     */
    public function registrarPermisos($permisos, $idUsuarioCreate, $idRolUsuarioCreate, $controlador, $logu, $logs) {
        if(isset($permisos) && $permisos != null) {
            foreach($permisos as $idPermiso) {
                try {
                    $model = new RolUsuarioPermiso();
                    $model->idRolUsuario = $this->idRolUsuario;
                    $model->idPermiso = $idPermiso;

                    $model->registrar($idUsuarioCreate, $idRolUsuarioCreate);

                    $mensaje =  "SE REGISTRÓ EL ROL DE USUARIO PERMISO CON ID: " . $model->idRolUsuarioPermiso;
                    $logu->registrar($idUsuarioCreate, $idRolUsuarioCreate, $mensaje, 1);
                    $logs->registrar($idUsuarioCreate, $idRolUsuarioCreate, "DEBUG", $controlador, $mensaje);
                } catch(\Exception $ex) {
                    $mensaje =  "NO SE REGISTRÓ EL ROL DE USUARIO PERMISO CON idRolUsuario: " . $model->idRolUsuario . " E idPermiso: " .$model->idPermiso;
                    $logu->registrar($idUsuarioCreate, $idRolUsuarioCreate, $mensaje, 0);
                    $logs->registrar($idUsuarioCreate, $idRolUsuarioCreate, "ERROR", $controlador, $ex->getMessage());

                    throw $ex;
                }
            }
        }
    }

    /**
     * Método que actualiza un rol de usuario.
     *
     * @param $idUsuarioRecord
     * @param $idRolUsuarioRecord
     * @throws \Exception
     */
    public function actualizar($idUsuarioRecord, $idRolUsuarioRecord) {
        $this->recordDate = new Expression('NOW()');
        $this->idUsuarioRecord = $idUsuarioRecord;
        $this->idRolUsuarioRecord = $idRolUsuarioRecord;

        if(!$this->save(false)) {
            $error = print_r($this->getErrors(), true);
            throw new \Exception($error);
        }
    }

    /**
     * Método que elimina un rol de usuario.
     *
     * @throws \Exception
     */
    public function eliminar($idUsuarioDelete, $idRolUsuarioDelete) {
        if(!$this->delete()) {
            $error = print_r($this->getErrors(), true);
            throw new \Exception($error);
        }
    }

    /**
     * Método que devuelve los permisos que están relacionadas con el rol de usuario.
     *
     * @return \yii\db\ActiveQuery
     */
    public function getRolUsuarioPermisos() {
        return $this->hasMany(RolUsuarioPermiso::className(), ['idRolUsuario' => 'idRolUsuario']);
    }

    /**
     * Método que devuelve los usuarios relacionados a un idRolUsuario.
     */
    public function getUsuarios() {
        return $this->hasMany(Usuario::className(), ['idRolUsuario' => 'idRolUsuario']);
    }

    /**
     * Método que verifica si existe un rol de usaurio de acuerdo a los parámetros proporcionados.
     *
     * @param $rolUsuario
     * @param $idRolUsuario
     * @return bool
     */
    public static function verificarExistencia($rolUsuario, $idRolUsuario = null) {
        $registro = null;

        if($idRolUsuario == null)
            $registro = static::find()->where(['rolUsuario' => $rolUsuario])->one();
        else
            $registro = static::find()->where(['rolUsuario' => $rolUsuario])
                ->andWhere(['!=', 'idRolUsuario', $idRolUsuario])
                ->one();

        if($registro != null)
            return true;

        return false;
    }
}


?>