<?php
/*
 * Sustentante.php
 * Springfield
 * Copyright ©: 2022 Integrasoft. Todos los derechos reservados.
 * Desarrollador(es):
 *      I.S.C Julio Alberto Ocampo Rangel.
 *      L.I. Humberto Javier Flores Tirado.
 * Fecha de creación: 27/03/2022.
 * Fecha de modificación: 19/05/2022.
 * Descripción:
 */
namespace app\models\sustentante;

use PHPUnit\Framework\Exception;
use Yii;
use yii\db\ActiveRecord;
use yii\db\Expression;
use app\models\antecedenteacademico\AntecedenteAcademico;
use app\models\estado\Estado;
use app\models\estatussustentante\EstatusSustentante;
use app\models\motivonoconcluyo\MotivoNoConcluyo;
use app\models\niveleducativo\NivelEducativo;
use app\models\perfilprofesional\PerfilProfesional;
use app\models\sexo\Sexo;
use app\models\usuario\Usuario;
use app\models\sesion\UsuarioSesion;

/**
 * Clase modelo de la tabla "Sustentante".
 *
 * @property int $idSustentante
 * @property int $idSexo
 * @property int $idEstado
 * @property int $idPerfilProfesional
 * @property int $idMotivoNoConcluyo
 * @property int $idAntecedenteAcademico
 * @property int $idUsuario
 * @property int $idEstadoRadica
 * @property int $idNivelEducativo
 * @property int $idEstatusSustentante
 * @property string $numeroControl
 * @property string $anioNumeroControl
 * @property string $mesNumeroControl
 * @property string $consecutivoNumeroControl
 * @property string $curp
 * @property string $fechaNacimiento
 * @property string $nacionalidad
 * @property string $lugarNacimiento
 * @property string $calleNumero
 * @property string $colonia
 * @property string $codigoPostal
 * @property string $municipio
 * @property string $telefonoCasa
 * @property string $telefonoOficina
 * @property string $celular
 * @property string $institucion
 * @property double $porcentajeCreditosCursados
 * @property string $otroMotivo
 * @property string $licenciatura
 * @property double $porcentajeCreditosObtenidos
 * @property string $personaAutorizada1
 * @property string $archivoPersonaAutorizada1
 * @property string $personaAutorizada2
 * @property string $archivoPersonaAutorizada2
 * @property string $archivoActaNacimiento
 * @property string $archivoCurp
 * @property string $archivoIne
 * @property string $archivoCertificado
 * @property string $archivoAntecedenteSecundaria
 * @property string $archivoCurriculum
 * @property string $archivoDoctoUltimoGradoEstudios
 * @property string $archivoCartaAutorizacion
 * @property string $observaciones
 * @property string $createDate
 * @property string $recordDate
 * @property string $deleteDate
 * @property int $idUsuarioCreate
 * @property int $idUsuarioRecord
 * @property int $idUsuarioDelete
 * @property int $idRolUsuarioCreate
 * @property int $idRolUsuarioRecord
 * @property int $idRolUsuarioDelete
 *
 * @property AntecedenteAcademico $antecedenteAcademico
 * @property Estado $estado
 * @property Estado $estadoRadica
 * @property EstatusSustentante $estatusSustentante
 * @property MotivoNoConcluyo $motivoNoConcluyo
 * @property NivelEducativo $nivelEducativo
 * @property PerfilProfesional $perfilProfesional
 * @property Sexo $sexo
 * @property Usuario $usuario
 */
class Sustentante extends ActiveRecord
{
    public $idAreaConocimiento;
    public $contrasenia = "";
    public $verificarContrasenia = "";
    public $nombre = "";
    public $primerApellido = "";
    public $segundoApellido = "";
    public $correoElectronico = "";
    public $idSede;

    public $archivoPersonaAutorizada1Upload;
    public $archivoPersonaAutorizada2Upload;
    public $archivoActaNacimientoUpload;
    public $archivoCurpUpload;
    public $archivoIneUpload;
    public $archivoCertificadoUpload;
    public $archivoAntecedenteSecundariaUpload;
    public $archivoCurriculumUpload;
    public $archivoDoctoUltimoGradoEstudiosUpload;
    public $archivoCartaAutorizacionUpload;

    /**
     * Método que devuelve la tabla de la que es modelo la clase.
     */
    public static function tableName()
    {
        return 'Sustentante';
    }

    /**
     * Método que define las validaciones de los atributos.
     */
    public function rules()
    {
        return [
            [['nombre','primerApellido','segundoApellido','correoElectronico','contrasenia','verificarContrasenia', 'idSexo', 'idEstado', 'idEstadoRadica', 'idNivelEducativo', 'idAntecedenteAcademico', 'idSede', 'curp', 'fechaNacimiento', 'nacionalidad', 'lugarNacimiento', 'calleNumero', 'colonia', 'codigoPostal', 'municipio', 'celular', 'createDate', 'recordDate', 'idUsuarioCreate', 'idUsuarioRecord', 'idRolUsuarioCreate', 'idRolUsuarioRecord'], 'required', 'message' => 'Es obligatorio','on' => ['registrar','actualizar','validar']],
            [['idEstatusSustentante', 'idSexo', 'idEstado', 'idEstadoRadica', 'idPerfilProfesional', 'idNivelEducativo', 'idMotivoNoConcluyo', 'idAntecedenteAcademico', 'idUsuario', 'idEstatusUsuario', 'idUsuarioCreate', 'idUsuarioRecord', 'idUsuarioDelete', 'idRolUsuarioCreate', 'idRolUsuarioRecord', 'idRolUsuarioDelete'], 'integer', 'message' => 'Debe ser numérico','on' => ['registrar','actualizar','validar']],
            [['idEstatusSustentante', 'observaciones','idAreaConocimiento','fechaNacimiento', 'createDate', 'recordDate', 'deleteDate'], 'safe','on' => ['registrar','actualizar','validar']],
            [['anioNumeroControl', 'consecutivoNumeroControl', 'porcentajeCreditosCursados', 'porcentajeCreditosObtenidos'], 'number', 'message' => 'Debe ser númerico','on' => ['registrar','actualizar','validar']],
            [['nacionalidad', 'lugarNacimiento', 'colonia'], 'string', 'max' => 50, 'message' => 'Máximo 50 caracteres','on' => ['registrar','actualizar','validar']],
            [['numeroControl'], 'string', 'max' => 20, 'message' => 'Máximo 20 caracteres','on' => ['registrar','actualizar','validar']],
            [['mesNumeroControl'], 'string', 'max' => 2, 'message' => 'Máximo 2 caracteres','on' => ['registrar','actualizar','validar']],
            [['curp'], 'string', 'max' => 18, 'message' => 'Máximo 18 caracteres','on' => ['registrar','actualizar','validar']],
            [['calleNumero', 'municipio', 'institucion', 'licenciatura', 'personaAutorizada1', 'personaAutorizada2', 'archivoPersonaAutorizada1', 'archivoPersonaAutorizada2', 'archivoActaNacimiento', 'archivoCurp', 'archivoIne', 'archivoCertificado', 'archivoAntecedenteSecundaria', 'archivoCurriculum', 'archivoDoctoUltimoGradoEstudios'], 'string', 'max' => 100, 'message' => 'Máximo 100 caracteres'],
            [['codigoPostal'], 'string', 'max' => 5, 'message' => 'Máximo 5 caracteres','on' => ['registrar','actualizar','validar']],
            [['telefonoCasa', 'telefonoOficina', 'celular'], 'string', 'max' => 10, 'message' => 'Máximo 10 caracteres','on' => ['registrar','actualizar','validar']],
            [['otroMotivo'], 'string', 'max' => 200, 'message' => 'Máximo 200 caracteres','on' => ['registrar','actualizar','validar']],
            
            [['archivoActaNacimientoUpload',
                'archivoCurpUpload','archivoIneUpload','archivoCertificadoUpload'], 'file',
                'skipOnEmpty' => false,
                'uploadRequired' => 'Es obligatorio',
                'extensions' => 'jpg, png, pdf',
                'maxFiles' => 1,
                'wrongExtension' => 'Debe ser un archivo en formato jpg/png/pdf',
                'mimeTypes' => 'image/jpeg, image/png, application/pdf',
                'maxSize' => 102400000,
                'tooBig' => 'Debe ser menor o igual a 100mb',
                'on' => 'registrar'
            ],

            [['archivoPersonaAutorizada1Upload','archivoPersonaAutorizada2Upload','archivoAntecedenteSecundariaUpload','archivoCurriculumUpload','archivoDoctoUltimoGradoEstudiosUpload', 'archivoCartaAutorizacionUpload'], 
                'file',
                'skipOnEmpty' => true,
                'maxFiles' => 1,
                'extensions' => 'jpg, png, pdf',
                'wrongExtension' => 'Debe ser un archivo en formato jpg/png/pdf',
                'mimeTypes' => 'image/jpeg, image/png, application/pdf',
                'maxSize' => 102400000,
                'tooBig' => 'Debe ser menor o igual a 100mb',
                'on' => 'registrar'
            ],

            ['archivoAntecedenteSecundariaUpload', 'required', 'when' => function($model) {
                if($model->idNivelEducativo == 1)
                    return true;
            }, 
            'whenClient' => "function (attribute, value) {
                return $('#sustentante-idniveleducativo').val() == 1;
            }",
            'message' => 'Es obligatorio', 'on' => 'registrar'],

            [['archivoCurriculumUpload','archivoDoctoUltimoGradoEstudiosUpload'], 'required', 'when' => function($model) {
                if($model->idNivelEducativo == 2)
                    return true;
            }, 
            'whenClient' => "function (attribute, value) {
                return $('#sustentante-idniveleducativo').val() == 2;
            }",
            'message' => 'Es obligatorio', 'on' => 'registrar'],

            ['archivoCartaAutorizacionUpload', 'required', 'when' => function($model) {
                $anioActual = date("Y");
                if(!empty($model->fechaNacimiento)){
                    $fecha = explode('-',$model->fechaNacimiento);
                    if(($anioActual - $fecha[0]) < 18) {
                        return true;
                    }
                }                
            }, 
            'whenClient' => "function (attribute, value) {
                const date = new Date();
                const anioActual = date.getFullYear();
                const fechaNacimiento = $('#sustentante-fechanacimiento').val().split('-');
                const anioFechaNacimiento = fechaNacimiento[0];
                var result = anioActual - anioFechaNacimiento;
                if(result < 18)
                    return true;
            }",
            'message' => 'Es obligatorio', 'on' => 'registrar'],

            ['observaciones', 'required', 'when' => function($model) {
                if($model->idEstatusSustentante == 2)
                    return true;
            }, 
            'whenClient' => "function (attribute, value) {
                return $('#sustentante-idestatussustentante').val() == 2;
            }",
            'message' => 'Es obligatorio', 'on' => 'validar'],
            
            ['correoElectronico', 'rulesVerificarExistencia','on' => ['registrar','actualizar','validar']],
            ['contrasenia','required','message' => 'Es obligatoria','on' => ['registrar','actualizar','validar']],
            ['contrasenia', 'match', 'pattern' => "/^.{8,50}$/", 'message' => 'Debe tener entre 8 y 50 caracteres','on' => ['registrar','actualizar','validar']],
            ['contrasenia', 'rulesVerificarContraseniaValida','on' => ['registrar','actualizar','validar']],
            ['verificarContrasenia', 'required', 'when' => function($model) {
                if(!empty($model->contrasenia))
                    return true;
            }, 'message' => 'Es obligatoria','on' => ['registrar','actualizar','validar']],
            ['verificarContrasenia', 'compare', 'compareAttribute' => 'contrasenia', 'when' => function($model) {
                if(!empty($model->contrasenia))
                    return true;
            }, 'message' => 'No coincide la contraseña y su confirmación','on' => ['registrar','actualizar','validar']],
            ['fechaNacimiento', 'required', 'when' => function($model) {
                $hoy = date("Y-m-d");

                if($model->fechaNacimiento >= $hoy) {
                    return true;
                }
            }, 'message' => 'Debe ser menor a la fecha actual','on' => ['registrar','actualizar','validar']]
        ];
    }

    /**
     * Método que define las etiquetas de los atributos.
     */
    public function attributeLabels()
    {
        return [
            'idSustentante' => 'ID',
            'idSexo' => 'Sexo',
            'idEstado' => 'Entidad federativa',
            'idEstadoRadica' => 'Entidad federativa donde radica',
            'idPerfilProfesional' => 'Perfil profesional',
            'idMotivoNoConcluyo' => 'Motivo por el cual no concluyo sus estudios o no se tituló',
            'idAntecedenteAcademico' => 'Antecedente académico',
            'idUsuario' => 'Usuario',
            'idNivelEducativo' => 'Nivel educativo',
            'idAreaConocimiento' => 'Área de conocimiento',
            'idEstatusSustentante' => 'Estatus',
            'idSede' => 'Sede',
            'numeroControl' => 'Número de control',
            'anioNumeroControl' => 'Año de núumero de control',
            'mesNumeroControl' => 'Mes de número de control',
            'nombre' => 'Nombre(s)',
            'primerApellido' => 'Primer apellido',
            'segundoApellido' => 'Segundo apellido',
            'curp' => 'CURP',
            'fechaNacimiento' => 'Fecha nacimiento',
            'nacionalidad' => 'Nacionalidad',
            'lugarNacimiento' => 'Lugar de nacimiento',
            'calleNumero' => 'Calle y número',
            'colonia' => 'Colonia',
            'codigoPostal' => 'Código postal',
            'municipio' => 'Delegación/Municipio',
            'telefonoCasa' => 'Teléfono de casa',
            'telefonoOficina' => 'Teléfono de oficina',
            'celular' => 'Celular',
            'perfilProfesionalAcreditar' => 'Perfil profesional que desea acreditar',
            'institucion' => 'Institución educativa donde realizó los estudios que desea acreditar',
            'porcentajeCreditosCursados' => 'Porcentaje de créditos en la carrera o licenciatura cursada',
            'otroMotivo' => 'Otro Motivo',
            'licenciatura' => 'Licenciatura en ',
            'porcentajeCreditosObtenidos' => 'Porcentaje de avance de créditos obtenidos',
            'personaAutorizada1' => 'Persona autorizada 1',
            'archivoPersonaAutorizada1' => 'Identificación de persona autorizada 1',
            'personaAutorizada2' => 'Persona autorizada 2',
            'archivoPersonaAutorizada2' => 'Identificación de persona autorizada 2',
            'archivoActaNacimiento' => 'Archivo acta de nacimiento',
            'archivoCurp' => 'Archivo CURP',
            'archivoIne' => 'Archivo INE',
            'archivoCertificado' => 'Archivo certificado',
            'archivoAntecedenteSecundaria' => 'Archivo antecedente secundaria',
            'archivoCurriculum' => 'Archivo currículum',
            'archivoDoctoUltimoGradoEstudios' => 'Archivo documento de último grado de estudios',
            'observaciones' => 'Observaciones',
            'correoElectronico' => 'Usuario (Correo electrónico)',
            'contrasenia' => 'Contraseña',
            'verificarContrasenia' => 'Confirma la contraseña',
            'archivoCurriculum' => 'Archivo currículum',
            'archivoCartaAutorizacion' => 'Archivo carta de autorización',
            'archivoPersonaAutorizada1Upload' => 'Persona autorizada 1',
            'archivoPersonaAutorizada2Upload' => 'Persona autorizada 2',
            'archivoActaNacimientoUpload' => 'Archivo acta de nacimiento',
            'archivoCurpUpload' => 'Archivo CURP',
            'archivoIneUpload' => 'Archivo INE',
            'archivoCertificadoUpload' => 'Archivo certificado',
            'archivoAntecedenteSecundariaUpload' => 'Archivo antecedente secundaria',
            'archivoCurriculumUpload' => 'Archivo currículum',
            'archivoDoctoUltimoGradoEstudiosUpload' => 'Archivo documento de último grado de estudios',
            'archivoCartaAutorizacionUpload' => 'Carta de autorización',
            'createDate' => 'Fecha de creación',
            'recordDate' => 'Fecha de modificación',
            'deleteDate' => 'Fecha de eliminación',

        ];
    }

    /**
     * Método que registrar un rol de usuario.
     *
     * @param $idUsuarioCreate
     * @param $idRolUsuarioCreate
     * @throws \Exception
     */
    public function registrar($idUsuarioCreate, $idRolUsuarioCreate) {
        $this->createDate = new Expression('NOW()');
        $this->recordDate = new Expression('NOW()');
        $this->idUsuarioCreate = $idUsuarioCreate;
        $this->idUsuarioRecord = $idUsuarioCreate;
        $this->idRolUsuarioCreate = $idRolUsuarioCreate;
        $this->idRolUsuarioRecord = $idRolUsuarioCreate;

        if(!$this->save(false)) {
            $error = print_r($this->getErrors(), true);
            throw new \Exception($error);
        }
        
        $this->guardarArchivo("archivoPersonaAutorizada1",$this->archivoPersonaAutorizada1Upload,Yii::$app->params['ruta-archivos-sustentante'],Yii::$app->params['nombre-archivo-sustentante-persona-autorizada-1']);
        $this->guardarArchivo("archivoPersonaAutorizada2",$this->archivoPersonaAutorizada2Upload,Yii::$app->params['ruta-archivos-sustentante'],Yii::$app->params['nombre-archivo-sustentante-persona-autorizada-2']);
        $this->guardarArchivo("archivoActaNacimiento",$this->archivoActaNacimientoUpload,Yii::$app->params['ruta-archivos-sustentante'],Yii::$app->params['nombre-archivo-sustentante-acta-nacimiento']);
        $this->guardarArchivo("archivoCurp",$this->archivoCurpUpload,Yii::$app->params['ruta-archivos-sustentante'],Yii::$app->params['nombre-archivo-sustentante-curp']);
        $this->guardarArchivo("archivoIne",$this->archivoIneUpload,Yii::$app->params['ruta-archivos-sustentante'],Yii::$app->params['nombre-archivo-sustentante-ine']);
        $this->guardarArchivo("archivoCertificado",$this->archivoCertificadoUpload,Yii::$app->params['ruta-archivos-sustentante'],Yii::$app->params['nombre-archivo-sustentante-certificado']);
        $this->guardarArchivo("archivoAntecedenteSecundaria",$this->archivoAntecedenteSecundariaUpload,Yii::$app->params['ruta-archivos-sustentante'],Yii::$app->params['nombre-archivo-sustentante-antecedente-secundaria']);
        $this->guardarArchivo("archivoCurriculum",$this->archivoCurriculumUpload,Yii::$app->params['ruta-archivos-sustentante'],Yii::$app->params['nombre-archivo-sustentante-curriculum']);
        $this->guardarArchivo("archivoDoctoUltimoGradoEstudios",$this->archivoDoctoUltimoGradoEstudiosUpload,Yii::$app->params['ruta-archivos-sustentante'],Yii::$app->params['nombre-archivo-sustentante-docto-ultimo-grado-estudios']);
        $this->guardarArchivo("archivoCartaAutorizacion",$this->archivoCartaAutorizacionUpload,Yii::$app->params['ruta-archivos-sustentante'],Yii::$app->params['nombre-archivo-carta-aurotizacion']);
        
    }

    /**
     * Método que actualiza un rol de usuario.
     *
     * @param $idUsuarioRecord
     * @param $idRolUsuarioRecord
     * @throws \Exception
     */
    public function actualizar($idUsuarioRecord, $idRolUsuarioRecord) {
        $this->recordDate = new Expression('NOW()');
        $this->idUsuarioRecord = $idUsuarioRecord;
        $this->idRolUsuarioRecord = $idRolUsuarioRecord;

        if(!$this->save(false)) {
            $error = print_r($this->getErrors(), true);
            throw new \Exception($error);
        }

        $this->guardarArchivo("archivoPersonaAutorizada1",$this->archivoPersonaAutorizada1Upload,Yii::$app->params['ruta-archivos-sustentante'],Yii::$app->params['nombre-archivo-sustentante-persona-autorizada-1']);
        $this->guardarArchivo("archivoPersonaAutorizada2",$this->archivoPersonaAutorizada2Upload,Yii::$app->params['ruta-archivos-sustentante'],Yii::$app->params['nombre-archivo-sustentante-persona-autorizada-2']);
        $this->guardarArchivo("archivoActaNacimiento",$this->archivoActaNacimientoUpload,Yii::$app->params['ruta-archivos-sustentante'],Yii::$app->params['nombre-archivo-sustentante-acta-nacimiento']);
        $this->guardarArchivo("archivoCurp",$this->archivoCurpUpload,Yii::$app->params['ruta-archivos-sustentante'],Yii::$app->params['nombre-archivo-sustentante-curp']);
        $this->guardarArchivo("archivoIne",$this->archivoIneUpload,Yii::$app->params['ruta-archivos-sustentante'],Yii::$app->params['nombre-archivo-sustentante-ine']);
        $this->guardarArchivo("archivoCertificado",$this->archivoCertificadoUpload,Yii::$app->params['ruta-archivos-sustentante'],Yii::$app->params['nombre-archivo-sustentante-certificado']);
        $this->guardarArchivo("archivoAntecedenteSecundaria",$this->archivoAntecedenteSecundariaUpload,Yii::$app->params['ruta-archivos-sustentante'],Yii::$app->params['nombre-archivo-sustentante-antecedente-secundaria']);
        $this->guardarArchivo("archivoCurriculum",$this->archivoCurriculumUpload,Yii::$app->params['ruta-archivos-sustentante'],Yii::$app->params['nombre-archivo-sustentante-curriculum']);
        $this->guardarArchivo("archivoDoctoUltimoGradoEstudios",$this->archivoDoctoUltimoGradoEstudiosUpload,Yii::$app->params['ruta-archivos-sustentante'],Yii::$app->params['nombre-archivo-sustentante-docto-ultimo-grado-estudios']);
        $this->guardarArchivo("archivoCartaAutorizacion",$this->archivoCartaAutorizacionUpload,Yii::$app->params['ruta-archivos-sustentante'],Yii::$app->params['nombre-archivo-carta-aurotizacion']);
    }

    /**
     * Método que elimina un rol de usuario.
     *
     * @throws \Exception
     */
    public function eliminar($idUsuarioDelete, $idRolUsuarioDelete) {
        if(!$this->delete()) {
            $error = print_r($this->getErrors(), true);
            throw new \Exception($error);
        }
    }

    private function guardarArchivo($tipoArchivo, $archivoUpload, $ruta, $nombreArchivo) {
        if($archivoUpload != null)
        {
            $nombreArchivo = $this->idSustentante . "_" . $nombreArchivo .
            "." . $archivoUpload->extension;
            $archivo = Yii::$app->basePath . "/" . $ruta . $nombreArchivo;
            $this->$tipoArchivo = $nombreArchivo;

            if(!$this->save(false))
            {
                $error = print_r($this->getErrors(), true);
                throw new \Exception($error);
            }
            
            $archivoUpload->saveAs($archivo);
        }
    }

    /**
     * Método para las rules que verifica si la contraseña es válida o no.
     *
     * @param $attribute
     * @param $params
     * @return bool
     */
    public function rulesVerificarContraseniaValida($attribute, $params) {
        if(!UsuarioSesion::verificarContraseniaValida($this->contrasenia)) {
            $this->addError($attribute, "La contraseña debe contener por lo menos dos letras, un número y un caracter especial.");
            return true;
        }
        return false;
    }

    /**
     * Método para las rules que verifica si la contraseña es válida o no.
     *
     * @param $attribute
     * @param $params
     * @return bool
     */
    public function rulesVerificarContraseniaValidaActualizar($attribute, $params) {
        if(!empty($this->contrasenia)) {
            if(!UsuarioSesion::verificarContraseniaValida($this->contrasenia)) {
                $this->addError($attribute, "La contraseña debe contener por lo menos dos letras, un número y un caracter especial");
                return true;
            }
        }

        return false;
    }

    /**
     * Método para las rules que verifica si ya existe un usuario con ese usuario.
     *
     * @param $attribute
     * @param $params
     * @return bool
     */
    public function rulesVerificarExistencia($attribute, $params) {
        if(Usuario::verificarExistencia($this->usuario->usuario, $this->idUsuario)) {
            return true;
        }

        return false;
    }

    /**
     * Método para las rules que verifica si ya existe un usuario con ese usuario.
     *
     * @param $attribute
     * @param $params
     * @return bool
     */
    public function rulesVerificarExistenciaCurp($attribute, $params) {
        if(Sustentante::verificarExistenciaCurp($this->curp, $this->idSustentante)) {
            return true;
        }

        return false;
    }

    /**
     * Método que verifica si existe un usuario de acuerdo a los parámetros proporcionados.
     *
     * @param $curp
     * @param $idSustentante
     * @return bool
     */
    public static function verificarExistenciaCurp($curp, $idSustentante = null) {
        $registro = null;

        if($idSustentante == null)
            $registro = static::find()->where(['curp' => $curp])->one();
        else
            $registro = static::find()->where(['curp' => $curp])
                ->andWhere(['!=', 'idSustentante', $idSustentante])
                ->one();

        if($registro != null)
            return true;

        return false;
    }

    /**
     * Método para verificar que la fecha de nacimiento sea menor a la actual.
     *
     * @param $fechaNacimiento
     * @return bool
     */
    public static function verificarFechaNacimientoMenorAFechaActual($fechaNacimiento) {
        $hoy = date("Y-m-d");

        if($fechaNacimiento >= $hoy)
            return true;

        return false;
    }

    /**
     * Método que obtiene las letras correspondientes al mes para agregar al número de control.
     *
     * @param $mes
     * @return string
     */
    public static function obtenerLetasDelMes($mes) {
        $letrasMes = "";

        switch($mes) {
            case 1:
                $letrasMes = "EN";
                break;

            case 2:
                $letrasMes = "FB";
                break;

            case 3:
                $letrasMes = "MZ";
                break;

            case 4:
                $letrasMes = "AB";
                break;

            case 5:
                $letrasMes = "MY";
                break;

            case 6:
                $letrasMes = "JN";
                break;

            case 7:
                $letrasMes = "JL";
                break;

            case 8:
                $letrasMes = "AG";
                break;

            case 9:
                $letrasMes = "ST";
                break;

            case 10:
                $letrasMes = "OT";
                break;

            case 11:
                $letrasMes = "NV";
                break;

            case 12:
                $letrasMes = "DC";
                break;
        }

        return $letrasMes;
    }

    /**
     * Método que genera el número de control.
     *
     * @param $anio
     * @return int|mixed
     */
    public static function generarConsecutivo($anio) {
        $maximo = Sustentante::find()->where(['anioNumeroControl' => $anio])->max('consecutivoNumeroControl');
        $consecutivo = 0;

        if($maximo == "")
            $consecutivo = 1;
        else
            $consecutivo = $maximo + 1;

        return $consecutivo;
    }

    /**
     * Método que da formato al número consecutivo.
     *
     * @param $consecutivo
     * @return string
     */
    public static function  formatearConsecutivo($consecutivo) {
        $consecutivoFormateado = "";

        if($consecutivo < 10)
            $consecutivoFormateado = '000' . $consecutivo;
        elseif($consecutivo < 100)
            $consecutivoFormateado = '00' . $consecutivo;
        elseif($consecutivo < 1000)
            $consecutivoFormateado = '0' . $consecutivo;

        return $consecutivoFormateado;
    }

    /**
     * @return ActiveQuery
     */
    public function getAntecedenteAcademico()
    {
        return $this->hasOne(AntecedenteAcademico::class, ['idAntecedenteAcademico' => 'idAntecedenteAcademico']);
    }

    /**
     * @return ActiveQuery
     */
    public function getEstado()
    {
        return $this->hasOne(Estado::class, ['idEstado' => 'idEstado']);
    }

    /**
     * @return ActiveQuery
     */
    public function getMotivoNoConcluyo()
    {
        return $this->hasOne(MotivoNoConcluyo::class, ['idMotivoNoConcluyo' => 'idMotivoNoConcluyo']);
    }

    /**
     * @return ActiveQuery
     */
    public function getPerfilProfesional()
    {
        return $this->hasOne(PerfilProfesional::class, ['idPerfilProfesional' => 'idPerfilProfesional']);
    }

    /**
     * @return ActiveQuery
     */
    public function getSexo()
    {
        return $this->hasOne(Sexo::class, ['idSexo' => 'idSexo']);
    }

    /**
     * @return ActiveQuery
     */
    public function getEstadoRadica()
    {
        return $this->hasOne(Estado::class, ['idEstado' => 'idEstado']);
    }

    /**
     * @return ActiveQuery
     */
    public function getNivelEducativo()
    {
        return $this->hasOne(NivelEducativo::class, ['idNivelEducativo' => 'idNivelEducativo']);
    }

    /**
     * @return ActiveQuery
     */
    public function getEstatusSustentante()
    {
        return $this->hasOne(EstatusSustentante::class, ['idEstatusSustentante' => 'idEstatusSustentante']);
    }

    /**
     * @return ActiveQuery
     */
    public function getUsuario()
    {
        return $this->hasOne(Usuario::class, ['idUsuario' => 'idUsuario']);
    }
}
