<?php
/*
 * Usuario.php
 * Springfield
 * Copyright ©: 2022 Integrasoft. Todos los derechos reservados.
 * Desarrollador(es):
 *      L.I. Humberto Javier Flores Tirado.
 * Fecha de creación: 25/03/2022.
 * Fecha de modificación: 17/05/2022.
 * Descripción:
 */

namespace app\models\usuario;

use Yii;
use yii\db\ActiveRecord;
use yii\db\Expression;

use app\models\rolusuario\RolUsuario;
use app\models\estado\Estado;
use app\models\sesion\EstatusUsuario;
use app\models\sesion\UsuarioSesion;
use app\models\permiso\Permiso;
use app\models\sede\Sede;

/**
 * Clase modelo de la tabla "Usuario".
 *
 * @property int $idUsuario
 * @property int $idRolUsuario
 * @property int $idEstado
 * @property int $idEstatusUsuario
 * @property int $idSede
 * @property int $idSesion
 * @property string $usuario
 * @property string $contrasenia
 * @property string $nombre
 * @property string $primerApellido
 * @property string $segundoApellido
 * @property string $createDate
 * @property string $recordDate
 * @property string $deleteDate
 * @property int $idUsuarioCreate
 * @property int $idUsuarioRecord
 * @property int $idUsuarioDelete
 * @property int $idRolUsuarioCreate
 * @property int $idRolUsuarioRecord
 * @property int $idRolUsuarioDelete
 *
 * @property RolUsuario $rolUsuario
 * @property EstatusUsuario $estatusUsuario
 * @property Sede $sede
 *
 * @property string $confirmaContrasenia
 */
class Usuario extends ActiveRecord {
    public $confirmaContrasenia;

    /**
     * Método que devuelve la tabla de la que es modelo la clase.
     */
    public static function tableName() {
        return 'Usuario';
    }

    /**
     * Método que define las validaciones de los atributos.
     */
    public function rules() {
        return [
            ['idRolUsuario', 'required', 'message' => 'Es obligatorio'],
            ['idRolUsuario', 'compare', 'compareValue' => 0, 'operator' => '>', 'message' => 'Es obligatorio'],
            ['idEstatusUsuario', 'required', 'message' => 'Es obligatorio'],
            ['idEstatusUsuario', 'compare', 'compareValue' => 0, 'operator' => '>', 'message' => 'Es obligatorio'],
            [['idEstado', 'idSede'], 'integer', 'message' => 'Debe ser numérico'],
            ['usuario','required','message' => 'Es obligatorio'],
            ['usuario','email','message' => 'Debe ser un correo electrónico válido'],
            ['usuario', 'rulesVerificarExistencia'],
            ['contrasenia','required','message' => 'Es obligatoria', 'on' => 'registrar'],
            ['contrasenia', 'match', 'pattern' => "/^.{8,50}$/", 'message' => 'Debe tener entre 8 y 50 caracteres', 'on' => 'registrar'],
            ['contrasenia', 'rulesVerificarContraseniaValida', 'on' => 'registrar'],
            ['contrasenia', 'rulesVerificarContraseniaValidaActualizar', 'on' => 'actualizar'],
            ['confirmaContrasenia', 'required', 'message' => 'Es obligatoria', 'on' => 'registrar'],
            //['confirmaContrasenia', 'compare', 'compareAttribute' => 'contrasenia', 'message' => 'No coincide la contraseña y su confirmación'],
            ['confirmaContrasenia', 'required', 'when' => function($model) {
                if(!empty($model->contrasenia))
                    return true;
            }, 'message' => 'Es obligatoria'],
            ['confirmaContrasenia', 'compare', 'compareAttribute' => 'contrasenia', 'when' => function($model) {
                if(!empty($model->contrasenia))
                    return true;
            }, 'message' => 'No coincide la contraseña y su confirmación'],

            ['nombre','required','message' => 'Es obligatorio'],
            ['nombre', 'match', 'pattern' => "/^.{3,100}$/", 'message' => 'Debe tener entre 3 y 50 caracteres'],
            ['primerApellido','required','message' => 'Es obligatorio'],
            ['primerApellido', 'match', 'pattern' => "/^.{3,100}$/", 'message' => 'Debe tener entre 3 y 50 caracteres'],
            ['segundoApellido', 'match', 'pattern' => "/^.{3,100}$/", 'message' => 'Debe tener entre 3 y 50 caracteres'],

            ['idEstado', 'required', 'when' => function($model) {
                if($model->idRolUsuario == 3)
                    return true;
            }, 'message' => 'Es obligatorio'],

            ['idSede', 'required', 'when' => function($model) {
                if($model->idRolUsuario == 3)
                    return true;
                elseif($model->idEstado < 1)
                    return true;
            }, 'message' => 'Es obligatorio'],

            /*
            [['idRolUsuario', 'idTipoUsuario', 'idEstatusUsuario', 'idSesion', 'usuario', 'nombre', 'apellidos', 'createDate', 'recordDate', 'idUsuarioCreate', 'idUsuarioRecord', 'idRolUsuarioCreate', 'idRolUsuarioRecord'], 'required'],
            [['idRolUsuario', 'idTipoUsuario', 'idEstatusUsuario', 'idSesion', 'idUsuarioCreate', 'idUsuarioRecord', 'idUsuarioDelete', 'idRolUsuarioCreate', 'idRolUsuarioRecord', 'idRolUsuarioDelete'], 'integer'],
            [['createDate', 'recordDate', 'deleteDate'], 'safe'],
            [['usuario', 'contrasenia', 'nombre', 'apellidos', 'correoElectronicoContacto'], 'string', 'max' => 50],
            [['idSesion'], 'unique'],

            [['idEstatusUsuario'], 'exist', 'skipOnError' => true, 'targetClass' => EstatusUsuario::className(), 'targetAttribute' => ['idEstatusUsuario' => 'idEstatusUsuario']],
            [['idRolUsuario'], 'exist', 'skipOnError' => true, 'targetClass' => RolUsuario::className(), 'targetAttribute' => ['idRolUsuario' => 'idRolUsuario']],
            [['idBanco'], 'exist', 'skipOnError' => true, 'targetClass' => Banco::className(), 'targetAttribute' => ['idBanco' => 'idBanco']],
            [['idPerfil'], 'exist', 'skipOnError' => true, 'targetClass' => Perfil::className(), 'targetAttribute' => ['idPerfil' => 'idPerfil']],
            [['idTipoUsuario'], 'exist', 'skipOnError' => true, 'targetClass' => TipoUsuario::className(), 'targetAttribute' => ['idTipoUsuario' => 'idTipoUsuario']],
            [['idSede'], 'exist', 'skipOnError' => true, 'targetClass' => Sede::class, 'targetAttribute' => ['idSede' => 'idSede']],
            */
        ];
    }

    /**
     * Método que define las etiquetas de los atributos.
     */
    public function attributeLabels() {
        return [
            'idUsuario' => 'ID',
            'idRolUsuario' => 'Rol de usuario',
            'idEstado' => 'Estado',
            'idEstatusUsuario' => 'Estatus de usuario',
            'idSede' => 'Sede',
            'idSesion' => 'Sesión',
            'usuario' => 'Usuario',
            'contrasenia' => 'Contraseña',
            'nombre' => 'Nombre(s)',
            'primerApellido' => 'Primer apellido',
            'segundoApellido' => 'Segundo apellido',
            'createDate' => 'Fecha de creación',
            'recordDate' => 'Fecha de modificación',
            'deleteDate' => 'Fecha de eliminación',

            'confirmaContrasenia' => 'Confirma contraseña',
        ];
    }

    /**
     * Método que registra un usuario.
     *
     * @param $idUsuarioCreate
     * @param $idRolUsuarioCreate
     * @throws \Exception
     */
    public function registrar($idUsuarioCreate, $idRolUsuarioCreate) {
        $this->idSesion = str_replace(".", "", microtime(true));
        $this->contrasenia = md5($this->contrasenia);
        $this->createDate = new Expression('NOW()');
        $this->recordDate = new Expression('NOW()');
        $this->idUsuarioCreate = $idUsuarioCreate;
        $this->idUsuarioRecord = $idUsuarioCreate;
        $this->idRolUsuarioCreate = $idRolUsuarioCreate;
        $this->idRolUsuarioRecord = $idRolUsuarioCreate;

        if(!$this->save(false)) {
            $error = print_r($this->getErrors(), true);
            throw new \Exception($error);
        }
    }

    /**
     * Método que actualiza un usuario.
     *
     * @param $idUsuarioRecord
     * @param $idRolUsuarioRecord
     * @throws \Exception
     */
    public function actualizar($idUsuarioRecord, $idRolUsuarioRecord) {
        $this->recordDate = new Expression('NOW()');
        $this->idUsuarioRecord = $idUsuarioRecord;
        $this->idRolUsuarioRecord = $idRolUsuarioRecord;

        if(!$this->save(false))
        {
            $error = print_r($this->getErrors(), true);
            throw new \Exception($error);
        }
    }

    /**
     * Método que elimina un usuario.
     *
     * @throws \Exception
     */
    public function eliminar($idUsuarioDelete, $idRolUsuarioDelete) {
        if(!$this->delete())
        {
            $error = print_r($this->getErrors(), true);
            throw new \Exception($error);
        }
    }

    /**
     * Método que cambia la contraseña de un usuario.
     *
     * @param $contrasnia
     * @param $idUsuarioRecord
     * @param $idRolUsuarioRecord
     * @throws \Exception
     */
    public function cambiarContrasenia($contrasenia, $idUsuarioRecord, $idRolUsuarioRecord) {
        $this->contrasenia = md5($contrasenia);
        $this->recordDate = new Expression('NOW()');
        $this->idUsuarioRecord = $idUsuarioRecord;
        $this->idRolUsuarioRecord = $idRolUsuarioRecord;

        if(!$this->save(false)) {
            $error = print_r($this->getErrors(), true);
            throw new \Exception($error);
        }
    }

    /**
     * Método que devuelve el rol del usuario.
     *
     * @return \yii\db\ActiveQuery
     */
    public function getRolUsuario() {
        return $this->hasOne(RolUsuario::className(), ['idRolUsuario' => 'idRolUsuario']);
    }

    /**
     * Método que devuelve el estado.
     *
     * @return \yii\db\ActiveQuery
     */
    public function getEstado() {
        return $this->hasOne(Estado::className(), ['idEstado' => 'idEstado']);
    }

    /**
     * Método que devuelve el estatus del usuario.
     *
     * @return \yii\db\ActiveQuery
     */
    public function getEstatusUsuario() {
        return $this->hasOne(EstatusUsuario::className(), ['idEstatusUsuario' => 'idEstatusUsuario']);
    }

    /**
     * Método que devuelve la sede del usuario.
     *
     * @return ActiveQuery
     */
    public function getSede() {
        return $this->hasOne(Sede::class, ['idSede' => 'idSede']);
    }

    /**
     * Método para las rules que verifica si la contraseña es válida o no.
     *
     * @param $attribute
     * @param $params
     * @return bool
     */
    public function rulesVerificarContraseniaValida($attribute, $params) {
        if(!UsuarioSesion::verificarContraseniaValida($this->contrasenia)) {
            $this->addError($attribute, "La contraseña debe contener por lo menos dos letras, un número y un caracter especial.");
            return true;
        }
        return false;
    }

    /**
     * Método para las rules que verifica si la contraseña es válida o no.
     *
     * @param $attribute
     * @param $params
     * @return bool
     */
    public function rulesVerificarContraseniaValidaActualizar($attribute, $params) {
        if(!empty($this->contrasenia)) {
            if(!UsuarioSesion::verificarContraseniaValida($this->contrasenia)) {
                $this->addError($attribute, "La contraseña debe contener por lo menos dos letras, un número y un caracter especial");
                return true;
            }
        }

        return false;
    }

    /**
     * Método para las rules que verifica si ya existe un usuario con ese usuario.
     *
     * @param $attribute
     * @param $params
     * @return bool
     */
    public function rulesVerificarExistencia($attribute, $params) {
        if(Usuario::verificarExistencia($this->usuario, $this->idUsuario)) {
            return true;
        }

        return false;
    }

    /**
     * Método que verifica si existe un usuario de acuerdo a los parámetros proporcionados.
     *
     * @param $usuario
     * @param $idUsuario
     * @return bool
     */
    public static function verificarExistencia($usuario, $idUsuario = null) {
        $registro = null;

        if($idUsuario == null)
            $registro = static::find()->where(['usuario' => $usuario])->one();
        else
            $registro = static::find()->where(['usuario' => $usuario])
                ->andWhere(['!=', 'idUsuario', $idUsuario])
                ->one();

        if($registro != null)
            return true;

        return false;
    }

    /**
     * Método que verifica si el usuario tiene acceso al permiso especificado.
     *
     * @param $permiso
     * @return bool
     */
    public function verificarPermisos($permiso){
        $registro = null;

        $registro = Permiso::find()
            ->innerJoin('RolUsuarioPermiso RUP', 'RUP.idPermiso = Permiso.idPermiso')
            ->innerJoin('Usuario US', 'US.idRolUsuario = RUP.idRolUsuario')
            ->where(['US.idUsuario' => $this->idUsuario, 'Permiso.permiso' => $permiso])
            ->one();

        if($registro != null)
            return true;


        return false;
    }
}
